# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_resource_allocation(total_budget):
    """Optimize resource allocation to minimize casualties within budget constraints."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="resource_allocation")
    
    # Data from the problem context
    killed = [10, 5, 20]
    injured = [50, 30, 100]
    costs = [2000, 1500, 2500]
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(killed) == len(injured) == len(costs) == 3, "Array length mismatch"
    safe_range = range(min(len(killed), len(injured), len(costs)))  # Safe indexing
    
    # 2. VARIABLES
    x = {i: mdl.continuous_var(name=f"x_{i}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Minimize the total number of casualties
    objective = mdl.sum((killed[i] + injured[i]) * x[i] for i in safe_range)
    mdl.minimize(objective)
    
    # 4. CONSTRAINTS
    # Budget constraint
    budget_constraint = mdl.sum(costs[i] * x[i] for i in safe_range)
    mdl.add_constraint(budget_constraint <= total_budget, ctname="budget_constraint")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 1e-6:
                print(f"x[{i}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Example usage
total_budget = 1000000  # Example budget
optimize_resource_allocation(total_budget)