# Complete Optimization Problem and Solution: pets_1

## 1. Problem Context and Goals

### Context  
The university is focused on optimizing the allocation of student advisors to students who own pets. The primary aim is to ensure that each advisor's workload is balanced, considering both the number of students and the total weight of pets assigned to them. The decision-making process involves assigning students to advisors, where the number of students assigned is an integer value, and the total pet weight assigned is a continuous value. The operational goal is to minimize the maximum workload of any advisor, which is determined by the sum of the number of students and the total pet weight assigned to each advisor. The business configuration includes a workload coefficient, set at 1.0, which ensures that both student numbers and pet weights are equally considered in workload calculations. This setup allows for precise operational decision-making that aligns with linear optimization formulations, avoiding any nonlinear relationships such as variable products or divisions. The constraints are based on the maximum number of students and total pet weight each advisor can handle, ensuring resource limitations are respected.

### Goals  
The optimization goal is to minimize the maximum workload of any advisor. This involves reducing the highest sum of students and pet weight assigned to any single advisor. Success in this optimization is measured by achieving a balanced distribution of workloads across all advisors, ensuring no advisor is overburdened. The linear optimization goal is clearly defined by focusing on minimizing the maximum workload, which is a straightforward linear objective.

## 2. Constraints    

The constraints for this optimization problem are designed to ensure that each advisor's workload remains within manageable limits. Specifically, the constraints include:

- Each advisor can handle a maximum number of students, which is defined by the advisor's capacity.
- Each advisor can also handle a maximum total pet weight, ensuring that the pet weight assigned does not exceed this limit.
- Each student must be assigned to exactly one advisor, ensuring that all students are accounted for in the allocation process.

These constraints are expressed in business terms that naturally lead to linear mathematical forms, ensuring that the optimization problem remains linear and manageable.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 2 Database Schema
-- Objective: Schema changes include creating new tables for missing optimization data, modifying existing tables to improve mapping adequacy, and updating business configuration logic for scalar parameters and formulas.

CREATE TABLE Has_Pet (
  StuID INTEGER,
  PetID INTEGER,
  AdvisorID INTEGER
);

CREATE TABLE AdvisorConstraints (
  AdvisorID INTEGER,
  MaxStudents INTEGER,
  MaxWeight FLOAT
);

CREATE TABLE PetWeightAssignment (
  AdvisorID INTEGER,
  TotalPetWeight FLOAT
);
```

### Data Dictionary  
The data dictionary provides a comprehensive mapping of tables and columns to their business purposes and optimization roles:

- **Has_Pet**: This table maps students to their pets and advisors. It plays a crucial role in decision variables, where each student (StuID) is uniquely identified and linked to their respective pet (PetID) and advisor (AdvisorID).

- **AdvisorConstraints**: This table stores the constraints for each advisor, detailing the maximum number of students (MaxStudents) and the maximum total pet weight (MaxWeight) that each advisor can handle. These constraints are critical for ensuring that the optimization respects the resource limitations of each advisor.

- **PetWeightAssignment**: This table records the total weight of pets assigned to each advisor. It is used to calculate the advisor's workload, ensuring that the total pet weight assigned does not exceed the advisor's capacity.

### Current Stored Values  
```sql
-- Iteration 2 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on typical university advisor workloads and average pet weights, ensuring that constraints are neither too tight nor too loose, allowing for a balanced distribution of students and pets.

-- Realistic data for Has_Pet
INSERT INTO Has_Pet (StuID, PetID, AdvisorID) VALUES (1, 101, 201);
INSERT INTO Has_Pet (StuID, PetID, AdvisorID) VALUES (2, 102, 202);
INSERT INTO Has_Pet (StuID, PetID, AdvisorID) VALUES (3, 103, 203);

-- Realistic data for AdvisorConstraints
INSERT INTO AdvisorConstraints (AdvisorID, MaxStudents, MaxWeight) VALUES (201, 5, 100.0);
INSERT INTO AdvisorConstraints (AdvisorID, MaxStudents, MaxWeight) VALUES (202, 6, 120.0);
INSERT INTO AdvisorConstraints (AdvisorID, MaxStudents, MaxWeight) VALUES (203, 7, 150.0);

-- Realistic data for PetWeightAssignment
INSERT INTO PetWeightAssignment (AdvisorID, TotalPetWeight) VALUES (201, 50.0);
INSERT INTO PetWeightAssignment (AdvisorID, TotalPetWeight) VALUES (202, 60.0);
INSERT INTO PetWeightAssignment (AdvisorID, TotalPetWeight) VALUES (203, 70.0);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- \( x_{ij} \): Binary variable indicating whether student \( i \) is assigned to advisor \( j \). \( x_{ij} = 1 \) if student \( i \) is assigned to advisor \( j \), otherwise \( x_{ij} = 0 \).
- \( w_j \): Continuous variable representing the total pet weight assigned to advisor \( j \).

#### Objective Function
Minimize the maximum workload of any advisor, defined as the sum of the number of students and the total pet weight assigned to each advisor:

\[
\min \max_{j} \left( \sum_{i} x_{ij} + w_j \right)
\]

To linearize the objective function, introduce an auxiliary variable \( M \) representing the maximum workload:

\[
\min M
\]

Subject to:

\[
M \geq \sum_{i} x_{ij} + w_j \quad \forall j
\]

#### Constraints
1. **Student Assignment Constraint**: Each student must be assigned to exactly one advisor.

\[
\sum_{j} x_{ij} = 1 \quad \forall i
\]

2. **Advisor Capacity Constraints**:
   - Maximum number of students per advisor:

\[
\sum_{i} x_{ij} \leq \text{MaxStudents}_j \quad \forall j
\]

   - Maximum total pet weight per advisor:

\[
w_j \leq \text{MaxWeight}_j \quad \forall j
\]

3. **Pet Weight Assignment**: Ensure that the total pet weight assigned to each advisor does not exceed the advisor's capacity. This is implicitly handled by the constraints on \( w_j \).

#### Data Source Verification
- \( \text{MaxStudents}_j \) and \( \text{MaxWeight}_j \) are sourced from the `AdvisorConstraints` table.
- The assignment of students to advisors \( x_{ij} \) is based on the `Has_Pet` table.
- The total pet weight \( w_j \) is initially sourced from the `PetWeightAssignment` table but is adjusted dynamically in the optimization process.

This formulation ensures that the optimization problem is linear and respects all constraints and coefficients as specified in the business problem description.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_advisor_workload():
    """Optimize the allocation of student advisors to minimize maximum workload."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("advisor_workload_optimization")
    
    # Data from the database
    students = [1, 2, 3]
    advisors = [201, 202, 203]
    max_students = {201: 5, 202: 6, 203: 7}
    max_weights = {201: 100.0, 202: 120.0, 203: 150.0}
    pet_weights = {201: 50.0, 202: 60.0, 203: 70.0}
    
    # Validate data lengths
    assert len(max_students) == len(max_weights) == len(pet_weights) == len(advisors), "Array length mismatch"
    
    # 2. VARIABLES
    x = model.addVars(students, advisors, vtype=GRB.BINARY, name="x")
    w = model.addVars(advisors, vtype=GRB.CONTINUOUS, name="w", lb=0)
    M = model.addVar(vtype=GRB.CONTINUOUS, name="M", lb=0)
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(M, GRB.MINIMIZE)
    
    # 4. CONSTRAINTS
    
    # Maximum workload constraint
    model.addConstrs((M >= gp.quicksum(x[i, j] for i in students) + w[j] for j in advisors), name="max_workload")
    
    # Student assignment constraint
    model.addConstrs((gp.quicksum(x[i, j] for j in advisors) == 1 for i in students), name="student_assignment")
    
    # Advisor capacity constraints
    model.addConstrs((gp.quicksum(x[i, j] for i in students) <= max_students[j] for j in advisors), name="max_students")
    model.addConstrs((w[j] <= max_weights[j] for j in advisors), name="max_weight")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for j in advisors:
            assigned_students = [i for i in students if x[i, j].x > 0.5]
            print(f"Advisor {j} is assigned students: {assigned_students} with total pet weight: {w[j].x:.2f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_advisor_workload()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 1.0
**Execution Time**: 0.57 seconds
**Reliability**: high
**Analysis**: Gurobipy found an optimal solution with an objective value of 1.0, which is consistent with the expected linearization of the problem. The execution time was efficient, indicating a reliable performance.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_advisor_allocation():
    """Optimize the allocation of student advisors to students with pets."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="pets_1")
    
    # Data from the database
    students = [1, 2, 3]
    advisors = [201, 202, 203]
    max_students = {201: 5, 202: 6, 203: 7}
    max_weights = {201: 100.0, 202: 120.0, 203: 150.0}
    initial_weights = {201: 50.0, 202: 60.0, 203: 70.0}
    
    # Validate array lengths
    assert len(students) == 3, "Mismatch in number of students"
    assert len(advisors) == 3, "Mismatch in number of advisors"
    
    # Safe range for indexing
    safe_range_students = range(len(students))
    safe_range_advisors = range(len(advisors))
    
    # 2. VARIABLES
    x = {(i, j): mdl.binary_var(name=f"x_{i}_{j}") for i in safe_range_students for j in safe_range_advisors}
    w = {j: mdl.continuous_var(name=f"w_{j}", lb=0) for j in safe_range_advisors}
    M = mdl.continuous_var(name="M", lb=0)
    
    # 3. OBJECTIVE FUNCTION
    mdl.minimize(M)
    
    # 4. CONSTRAINTS
    
    # Maximum workload constraint
    for j in safe_range_advisors:
        mdl.add_constraint(M >= mdl.sum(x[i, j] for i in safe_range_students) + w[j], ctname=f"max_workload_{j}")
    
    # Student assignment constraint
    for i in safe_range_students:
        mdl.add_constraint(mdl.sum(x[i, j] for j in safe_range_advisors) == 1, ctname=f"student_assignment_{i}")
    
    # Advisor capacity constraints
    for j in safe_range_advisors:
        mdl.add_constraint(mdl.sum(x[i, j] for i in safe_range_students) <= max_students[advisors[j]], ctname=f"max_students_{j}")
        mdl.add_constraint(w[j] <= max_weights[advisors[j]], ctname=f"max_weight_{j}")
    
    # Initial pet weight assignment
    for j in safe_range_advisors:
        mdl.add_constraint(w[j] == initial_weights[advisors[j]], ctname=f"initial_weight_{j}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range_students:
            for j in safe_range_advisors:
                if solution.get_value(x[i, j]) > 1e-6:
                    print(f"Student {students[i]} assigned to Advisor {advisors[j]}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Run the optimization
optimize_advisor_allocation()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 70.0
**Execution Time**: 6.61 seconds
**Reliability**: medium
**Analysis**: DOCplex also found an optimal solution but with a significantly different objective value of 70.0. This discrepancy suggests a potential issue with the interpretation or implementation of the objective function or constraints.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def optimize_advisor_workload():
    """Optimize the allocation of student advisors to minimize maximum workload."""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Example data based on the problem description
    students = [1, 2, 3]
    advisors = [201, 202, 203]
    max_students = {201: 5, 202: 6, 203: 7}
    max_weights = {201: 100.0, 202: 120.0, 203: 150.0}
    initial_weights = {201: 50.0, 202: 60.0, 203: 70.0}
    
    # CRITICAL: Validate array lengths before indexing
    assert len(students) == 3, "Student data length mismatch"
    assert len(advisors) == 3, "Advisor data length mismatch"
    
    # 3. SETS
    model.I = pyo.Set(initialize=students)
    model.J = pyo.Set(initialize=advisors)
    
    # 4. PARAMETERS
    model.MaxStudents = pyo.Param(model.J, initialize=max_students)
    model.MaxWeight = pyo.Param(model.J, initialize=max_weights)
    
    # 5. VARIABLES
    model.x = pyo.Var(model.I, model.J, within=pyo.Binary)
    model.w = pyo.Var(model.J, within=pyo.NonNegativeReals)
    model.M = pyo.Var(within=pyo.NonNegativeReals)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return model.M
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)
    
    # 7. CONSTRAINTS
    
    # Maximum workload constraint
    def max_workload_rule(model, j):
        return model.M >= sum(model.x[i, j] for i in model.I) + model.w[j]
    model.max_workload_constraint = pyo.Constraint(model.J, rule=max_workload_rule)
    
    # Student assignment constraint
    def student_assignment_rule(model, i):
        return sum(model.x[i, j] for j in model.J) == 1
    model.student_assignment_constraint = pyo.Constraint(model.I, rule=student_assignment_rule)
    
    # Advisor capacity constraints
    def max_students_rule(model, j):
        return sum(model.x[i, j] for i in model.I) <= model.MaxStudents[j]
    model.max_students_constraint = pyo.Constraint(model.J, rule=max_students_rule)
    
    def max_weight_rule(model, j):
        return model.w[j] <= model.MaxWeight[j]
    model.max_weight_constraint = pyo.Constraint(model.J, rule=max_weight_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Optional: Set solver options
    solver.options['TimeLimit'] = 300  # 5 minutes
    solver.options['MIPGap'] = 0.01    # 1% gap
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    # Check solver status
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")
        
        # Extract variable values
        print("\nVariable values:")
        for j in model.J:
            workload = sum(pyo.value(model.x[i, j]) for i in model.I) + pyo.value(model.w[j])
            print(f"Advisor {j}: Workload = {workload:.3f}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Run the optimization
optimize_advisor_workload()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 1.0
**Execution Time**: 2.80 seconds
**Reliability**: high
**Analysis**: Pyomo's results align with Gurobipy, providing an optimal value of 1.0. The execution time was reasonable, supporting the reliability of the solution.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | OPTIMAL | 1.00 | 0.57s | N/A | N/A |
| Docplex | OPTIMAL | 70.00 | 6.61s | N/A | N/A |
| Pyomo | OPTIMAL | 1.00 | 2.80s | N/A | N/A |

### Solver Consistency Analysis
**Result**: Solvers produced inconsistent results
**Consistent Solvers**: gurobipy, pyomo
**Inconsistent Solvers**: docplex
**Potential Issues**:
- DOCplex may have a different interpretation of the objective function or constraints.
- There could be a data input or preprocessing error specific to DOCplex.
**Majority Vote Optimal Value**: 1.0

### Final Recommendation
**Recommended Optimal Value**: 1.0
**Confidence Level**: HIGH
**Preferred Solver(s)**: gurobipy
**Reasoning**: Gurobipy provided a consistent and reliable solution with the expected optimal value, and its performance was efficient.

### Business Interpretation
**Overall Strategy**: The optimal workload for any advisor is minimized to 1.0, indicating a balanced distribution of students and pet weights across advisors.
**Objective Value Meaning**: The optimal objective value of 1.0 indicates that the maximum workload for any advisor is minimized, ensuring equitable distribution of responsibilities.
**Resource Allocation Summary**: Advisors should be assigned students and pet weights such that no advisor exceeds the workload of 1.0, maintaining balance and efficiency.
**Implementation Recommendations**: Verify data inputs and constraints, particularly for DOCplex. Implement the solution using Gurobipy to ensure optimal resource allocation.