# Complete DOCPLEX implementation

from docplex.mp.model import Model

def poker_player_optimization():
    """Optimize prize money allocation to maximize player satisfaction."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="poker_player")
    
    # Data: Satisfaction coefficients and constraints
    player_ids = [1, 2, 3]
    satisfaction_coefficients = [1.1, 0.9, 1.3]
    total_budget = 24000
    max_prize_per_player = 10000
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(player_ids) == len(satisfaction_coefficients), "Array length mismatch"
    safe_range = range(len(player_ids))  # Safe indexing
    
    # 2. VARIABLES
    # Prize money allocation for each player
    prize_money = {i: mdl.continuous_var(name=f"x_{player_ids[i]}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total satisfaction
    objective = mdl.sum(satisfaction_coefficients[i] * prize_money[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total budget constraint
    total_allocation = mdl.sum(prize_money[i] for i in safe_range)
    mdl.add_constraint(total_allocation <= total_budget, ctname="total_budget")
    
    # Individual prize limit constraints
    for i in safe_range:
        mdl.add_constraint(prize_money[i] <= max_prize_per_player, ctname=f"max_prize_{player_ids[i]}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(prize_money[i])
            if value > 1e-6:
                print(f"Prize money for player {player_ids[i]}: {value:.2f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
poker_player_optimization()