# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def poker_player_optimization():
    """Optimize prize money allocation for poker players to maximize satisfaction."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("poker_player")
    
    # Data: Satisfaction coefficients and constraints
    player_ids = [1, 2, 3]
    coefficients = [1.1, 0.9, 1.3]
    total_budget = 24000
    max_prize_per_player = 10000
    
    # CRITICAL: Validate array lengths before loops
    assert len(player_ids) == len(coefficients), "Array length mismatch"
    
    # 2. VARIABLES
    # Prize money allocation variables for each player
    prize_money = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=0) 
                   for i in player_ids}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total satisfaction
    model.setObjective(gp.quicksum(coefficients[i-1] * prize_money[i] for i in player_ids), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total budget constraint
    model.addConstr(gp.quicksum(prize_money[i] for i in player_ids) <= total_budget, name="total_budget")
    
    # Individual prize limit constraints
    for i in player_ids:
        model.addConstr(prize_money[i] <= max_prize_per_player, name=f"max_prize_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in player_ids:
            if prize_money[i].x > 1e-6:
                print(f"Prize money for player {i}: {prize_money[i].x:.2f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
poker_player_optimization()