# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def product_catalog_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="product_catalog")
    
    # Exchange rates
    r_de = 0.85  # Example exchange rate from dollars to euros
    r_dp = 0.75  # Example exchange rate from dollars to pounds
    
    # Data from the database (hardcoded for this example)
    prices_dollars = [12.99, 18.49, 25.0]
    prices_euros = [11.04, 15.72, 21.25]
    prices_pounds = [9.74, 13.87, 18.75]
    min_prices_dollars = [11.99, 16.99, 22.5]
    min_prices_euros = [10.19, 14.44, 19.13]
    min_prices_pounds = [8.99, 12.74, 16.88]
    sales_volumes = [120, 180, 90]
    stock_availabilities = [150, 200, 100]
    
    # Validate array lengths
    assert len(prices_dollars) == len(prices_euros) == len(prices_pounds) == len(min_prices_dollars) == len(min_prices_euros) == len(min_prices_pounds) == len(sales_volumes) == len(stock_availabilities), "Array length mismatch"
    
    safe_range = range(len(prices_dollars))  # Safe indexing
    
    # 2. VARIABLES
    p_d = {i: mdl.continuous_var(name=f"p_d_{i}", lb=min_prices_dollars[i]) for i in safe_range}
    p_e = {i: mdl.continuous_var(name=f"p_e_{i}", lb=min_prices_euros[i]) for i in safe_range}
    p_p = {i: mdl.continuous_var(name=f"p_p_{i}", lb=min_prices_pounds[i]) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum((p_d[i] * sales_volumes[i] + p_e[i] * sales_volumes[i] / r_de + p_p[i] * sales_volumes[i] / r_dp) for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    for i in safe_range:
        # Stock availability constraint
        mdl.add_constraint(sales_volumes[i] <= stock_availabilities[i], ctname=f"stock_availability_{i}")
        
        # Price consistency constraints
        mdl.add_constraint(p_e[i] == p_d[i] * r_de, ctname=f"price_consistency_euros_{i}")
        mdl.add_constraint(p_p[i] == p_d[i] * r_dp, ctname=f"price_consistency_pounds_{i}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            print(f"Product {i}: Price in dollars = {solution.get_value(p_d[i]):.2f}, Price in euros = {solution.get_value(p_e[i]):.2f}, Price in pounds = {solution.get_value(p_p[i]):.2f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

# Run the optimization
product_catalog_optimization()