# Complete DOCPLEX implementation - Retry Attempt 1

from docplex.mp.model import Model

def optimize_program_share():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="program_share")
    
    # Data from the problem context
    channels = [1, 2, 3]
    programs = [101, 102, 103]
    share_in_percent = {
        (1, 101): 12.5,
        (1, 102): 18.0,
        (2, 103): 15.0,
        (2, 101): 10.0,
        (3, 102): 20.0
    }
    max_share_capacity = {
        1: 30.0,
        2: 25.0,
        3: 20.0
    }
    
    # Validate data consistency
    assert all((c, p) in share_in_percent for c in channels for p in programs if (c, p) in share_in_percent), "Data inconsistency detected"
    
    # Safe range for indexing
    safe_range = [(c, p) for c in channels for p in programs if (c, p) in share_in_percent]
    
    # 2. VARIABLES
    x = {(c, p): mdl.binary_var(name=f"x_{c}_{p}") for c, p in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(share_in_percent[c, p] * x[c, p] for c, p in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    # Channel Share Capacity Constraints
    for c in channels:
        mdl.add_constraint(
            mdl.sum(share_in_percent[c, p] * x[c, p] for p in programs if (c, p) in share_in_percent) <= max_share_capacity[c],
            ctname=f"capacity_{c}"
        )
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for c, p in safe_range:
            if solution.get_value(x[c, p]) > 0.5:  # Binary variable threshold
                print(f"Program {p} is broadcasted on Channel {c}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

# Execute the optimization
optimize_program_share()