# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_song_promotion(budget, max_songs):
    """Optimize song promotion to maximize sales increase within budget and song limit constraints."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="song_promotion")
    
    # Data from the problem statement
    potential_sales_increase = [1200.0, 1800.0, 2500.0]
    promotion_costs = [600.0, 900.0, 1200.0]
    n_songs = len(potential_sales_increase)
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(potential_sales_increase) == len(promotion_costs) == n_songs, "Array length mismatch"
    safe_range = range(n_songs)  # Safe indexing
    
    # 2. VARIABLES
    # Binary decision variables for each song
    x = {i: mdl.binary_var(name=f"x_{i}") for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total potential sales increase from promoting the songs
    objective = mdl.sum(potential_sales_increase[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Budget constraint
    total_cost = mdl.sum(promotion_costs[i] * x[i] for i in safe_range)
    mdl.add_constraint(total_cost <= budget, ctname="budget_constraint")
    
    # Promotion limit constraint
    total_promoted = mdl.sum(x[i] for i in safe_range)
    mdl.add_constraint(total_promoted <= max_songs, ctname="promotion_limit_constraint")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            if solution.get_value(x[i]) > 0.5:  # Binary variable, check if promoted
                print(f"Song {i+1} is promoted.")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Example usage
optimize_song_promotion(budget=1500, max_songs=2)