# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_song_promotion(budget, max_songs):
    """Optimize song promotion strategy to maximize sales increase."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("song_promotion")
    
    # Data from the database
    potential_sales_increase = [1200.0, 1800.0, 2500.0]
    promotion_costs = [600.0, 900.0, 1200.0]
    n_songs = len(potential_sales_increase)
    
    # CRITICAL: Validate array lengths before loops
    assert len(potential_sales_increase) == len(promotion_costs) == n_songs, "Array length mismatch"
    
    # 2. VARIABLES
    # Binary decision variables for each song
    x = {i: model.addVar(vtype=GRB.BINARY, name=f"x_{i}") for i in range(n_songs)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total potential sales increase
    model.setObjective(gp.quicksum(potential_sales_increase[i] * x[i] for i in range(n_songs)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Budget constraint
    model.addConstr(gp.quicksum(promotion_costs[i] * x[i] for i in range(n_songs)) <= budget, name="budget_constraint")
    
    # Promotion limit constraint
    model.addConstr(gp.quicksum(x[i] for i in range(n_songs)) <= max_songs, name="promotion_limit")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_songs):
            if x[i].x > 1e-6:
                print(f"Song {i+1} is promoted.")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Example usage
budget = 1500.0  # Example budget
max_songs = 2    # Example maximum number of songs to promote
optimize_song_promotion(budget, max_songs)