# Complete Optimization Problem and Solution: small_bank_1

## 1. Problem Context and Goals

### Context  
A small bank is focused on optimizing how customer funds are allocated between savings and checking accounts. The primary decision involves determining the amount of money each customer should allocate to their savings and checking accounts. The goal is to maximize the interest earned from these allocations. Each customer's savings and checking balances are treated as continuous decision variables. The bank aims to maximize the total interest earned by applying the respective interest rates to these balances. 

The bank operates under specific business configurations, which include maintaining a minimum required balance for both savings and checking accounts. These minimum balances are crucial for ensuring that customers can earn interest and manage daily transactions effectively. The bank also ensures that the total allocation of funds does not exceed the available funds for each customer. 

The operational parameters are aligned with a linear objective, focusing on maximizing the total interest earned. The bank's business logic is designed to ensure that all decisions and constraints are linear, avoiding any complex relationships such as variable products or divisions. The interest rates applied to savings and checking accounts serve as the coefficients in the optimization model, and the minimum balance requirements are used as constraint bounds.

### Goals  
The primary goal of the bank is to maximize the total interest earned from customer accounts. This involves optimizing the allocation of funds between savings and checking accounts for each customer. The success of this optimization is measured by the total interest earned, which is calculated by applying the respective interest rates to the savings and checking balances. The bank's objective is to achieve the highest possible total interest while adhering to the constraints of minimum balance requirements and total available funds. The optimization goal is clearly defined in linear terms, focusing on maximizing the interest without involving any complex mathematical operations.

## 2. Constraints    

The bank operates under several key constraints to ensure that the optimization problem remains linear and feasible:

- **Total Funds Constraint**: For each customer, the combined allocation to savings and checking accounts must not exceed the total funds available. This ensures that the bank does not allocate more funds than are available for each customer.

- **Minimum Savings Balance Constraint**: Each customer's savings account must maintain a minimum balance. This requirement ensures that customers can earn interest on their savings and aligns with typical banking policies.

- **Minimum Checking Balance Constraint**: Each customer's checking account must also maintain a minimum balance. This constraint ensures that customers have sufficient funds for daily transactions, which is a common practice in banking.

These constraints are described in business terms that naturally lead to linear mathematical forms, ensuring that the optimization problem remains straightforward and solvable using linear programming techniques.

## 3. Available Data  

### Database Schema  
```sql
-- Iteration 1 Database Schema
-- Objective: Schema changes include creating new tables for interest rates and minimum balances, and updating configuration logic for scalar parameters and formulas.

CREATE TABLE SAVINGS (
  balance FLOAT,
  interest_rate FLOAT
);

CREATE TABLE CHECKING (
  balance FLOAT,
  interest_rate FLOAT
);

CREATE TABLE CUSTOMER_FUNDS (
  total_funds FLOAT
);
```

### Data Dictionary  
The data dictionary provides a comprehensive overview of the tables and columns used in the optimization problem, highlighting their business purposes and roles in the optimization process:

- **SAVINGS Table**: This table stores information about the savings accounts for each customer. It includes the balance allocated to savings and the interest rate applied to these balances. The balance serves as a decision variable, while the interest rate acts as an objective coefficient.

- **CHECKING Table**: Similar to the savings table, this table contains data about the checking accounts for each customer. It includes the balance allocated to checking and the interest rate applied. The balance is a decision variable, and the interest rate is an objective coefficient.

- **CUSTOMER_FUNDS Table**: This table records the total funds available for allocation for each customer. The total funds serve as a constraint bound, ensuring that the allocation to savings and checking does not exceed the available funds.

### Current Stored Values  
```sql
-- Iteration 1 Realistic Data
-- Generated by triple expert (business + data + optimization)
-- Values were determined based on typical banking practices, ensuring that interest rates are competitive and balances are realistic for a small bank's customer base.

-- Realistic data for SAVINGS
INSERT INTO SAVINGS (balance, interest_rate) VALUES (1500.0, 0.012);
INSERT INTO SAVINGS (balance, interest_rate) VALUES (2500.0, 0.015);
INSERT INTO SAVINGS (balance, interest_rate) VALUES (3500.0, 0.018);

-- Realistic data for CHECKING
INSERT INTO CHECKING (balance, interest_rate) VALUES (800.0, 0.006);
INSERT INTO CHECKING (balance, interest_rate) VALUES (1200.0, 0.008);
INSERT INTO CHECKING (balance, interest_rate) VALUES (1800.0, 0.01);

-- Realistic data for CUSTOMER_FUNDS
INSERT INTO CUSTOMER_FUNDS (total_funds) VALUES (5000.0);
INSERT INTO CUSTOMER_FUNDS (total_funds) VALUES (10000.0);
INSERT INTO CUSTOMER_FUNDS (total_funds) VALUES (15000.0);
```

## 4. Mathematical Optimization Formulation

#### Decision Variables
- \( s_i \): Amount of funds allocated to the savings account for customer \( i \).
- \( c_i \): Amount of funds allocated to the checking account for customer \( i \).

#### Objective Function
Maximize the total interest earned from all customers:
\[
\text{Maximize } \sum_{i} (\text{SAVINGS.interest\_rate}_i \times s_i + \text{CHECKING.interest\_rate}_i \times c_i)
\]

#### Constraints
1. **Total Funds Constraint**: For each customer \( i \), the sum of allocations to savings and checking accounts must not exceed the total funds available.
   \[
   s_i + c_i \leq \text{CUSTOMER\_FUNDS.total\_funds}_i
   \]

2. **Minimum Savings Balance Constraint**: Each customer's savings account must maintain a minimum balance.
   \[
   s_i \geq \text{Minimum\_Savings\_Balance}_i
   \]

3. **Minimum Checking Balance Constraint**: Each customer's checking account must maintain a minimum balance.
   \[
   c_i \geq \text{Minimum\_Checking\_Balance}_i
   \]

Data Source Verification:
- Coefficients for the objective function:
  - \(\text{SAVINGS.interest\_rate}_i\) from SAVINGS.interest_rate
  - \(\text{CHECKING.interest\_rate}_i\) from CHECKING.interest_rate
- Constraint bounds:
  - \(\text{CUSTOMER\_FUNDS.total\_funds}_i\) from CUSTOMER_FUNDS.total_funds
  - \(\text{Minimum\_Savings\_Balance}_i\) and \(\text{Minimum\_Checking\_Balance}_i\) are assumed to be predefined constants based on bank policy (not explicitly provided in the data).

This linear model can be solved using standard linear programming techniques to determine the optimal allocation of funds for each customer, maximizing the total interest earned while adhering to all constraints.

## 5. Gurobipy Implementation

```python
# Complete GUROBIPY implementation - Retry Attempt 2

import gurobipy as gp
from gurobipy import GRB

def optimize_bank_funds():
    # 1. MODEL & DATA SETUP
    model = gp.Model("small_bank_1")
    
    # Data from the problem statement
    savings_interest_rates = [0.012, 0.015, 0.018]
    checking_interest_rates = [0.006, 0.008, 0.01]
    total_funds = [5000.0, 10000.0, 15000.0]
    min_savings_balance = [1000.0, 2000.0, 3000.0]  # Assumed minimum balances
    min_checking_balance = [500.0, 1000.0, 1500.0]  # Assumed minimum balances
    
    # Validate array lengths
    n_customers = len(total_funds)
    assert len(savings_interest_rates) == len(checking_interest_rates) == n_customers, "Array length mismatch"
    assert len(min_savings_balance) == len(min_checking_balance) == n_customers, "Array length mismatch"
    
    # 2. VARIABLES
    savings = model.addVars(n_customers, vtype=GRB.CONTINUOUS, name="savings", lb=0)
    checking = model.addVars(n_customers, vtype=GRB.CONTINUOUS, name="checking", lb=0)
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(
        gp.quicksum(savings_interest_rates[i] * savings[i] + checking_interest_rates[i] * checking[i] for i in range(n_customers)),
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    # Total Funds Constraint
    model.addConstrs(
        (savings[i] + checking[i] <= total_funds[i] for i in range(n_customers)),
        name="total_funds"
    )
    
    # Minimum Savings Balance Constraint
    model.addConstrs(
        (savings[i] >= min_savings_balance[i] for i in range(n_customers)),
        name="min_savings_balance"
    )
    
    # Minimum Checking Balance Constraint
    model.addConstrs(
        (checking[i] >= min_checking_balance[i] for i in range(n_customers)),
        name="min_checking_balance"
    )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_customers):
            print(f"Customer {i+1}: Savings = {savings[i].x:.2f}, Checking = {checking[i].x:.2f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_bank_funds()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 458.0
**Execution Time**: 0.15 seconds
**Reliability**: high
**Retry Attempt**: 2
**Analysis**: Gurobipy found an optimal solution quickly with a consistent optimal value across retries.

## 6. DOCplex Implementation

```python
# Complete DOCPLEX implementation - Retry Attempt 2

from docplex.mp.model import Model

def optimize_bank_funds():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="small_bank_1")
    
    # Data from the problem statement
    savings_interest_rates = [0.012, 0.015, 0.018]
    checking_interest_rates = [0.006, 0.008, 0.01]
    total_funds = [5000.0, 10000.0, 15000.0]
    min_savings_balance = [1000.0, 2000.0, 3000.0]  # Assumed minimum balances
    min_checking_balance = [500.0, 1000.0, 1500.0]  # Assumed minimum balances
    
    # Validate array lengths to prevent IndexError
    assert len(savings_interest_rates) == len(checking_interest_rates) == len(total_funds) == len(min_savings_balance) == len(min_checking_balance), "Array length mismatch"
    safe_range = range(len(total_funds))  # Safe indexing
    
    # 2. VARIABLES
    s = {i: mdl.continuous_var(name=f"s_{i}", lb=min_savings_balance[i]) for i in safe_range}
    c = {i: mdl.continuous_var(name=f"c_{i}", lb=min_checking_balance[i]) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(savings_interest_rates[i] * s[i] + checking_interest_rates[i] * c[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    for i in safe_range:
        # Total Funds Constraint
        mdl.add_constraint(s[i] + c[i] <= total_funds[i], ctname=f"total_funds_{i}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            savings_value = solution.get_value(s[i])
            checking_value = solution.get_value(c[i])
            print(f"Customer {i}: Savings = {savings_value:.2f}, Checking = {checking_value:.2f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Run the optimization
optimize_bank_funds()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 458.0
**Execution Time**: 1.70 seconds
**Reliability**: high
**Retry Attempt**: 2
**Analysis**: DOCplex also found an optimal solution with the same value as Gurobipy, indicating consistency.

## 7. Pyomo Implementation

```python
# Complete PYOMO implementation - Retry Attempt 2

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def small_bank_optimization():
    """Optimize fund allocation for a small bank using Pyomo"""
    
    # 1. MODEL CREATION
    model = pyo.ConcreteModel()
    
    # 2. DATA SETUP
    # Assuming 3 customers based on the provided data
    n_customers = 3
    
    # Interest rates and total funds from the database
    savings_interest_rates = [0.012, 0.015, 0.018]
    checking_interest_rates = [0.006, 0.008, 0.01]
    total_funds = [5000.0, 10000.0, 15000.0]
    
    # Minimum balance requirements (assumed constants)
    min_savings_balance = [1000.0, 1000.0, 1000.0]
    min_checking_balance = [500.0, 500.0, 500.0]
    
    # CRITICAL: Validate array lengths before indexing
    assert len(savings_interest_rates) == len(checking_interest_rates) == len(total_funds) == n_customers, "Array length mismatch"
    
    # 3. SETS
    model.I = pyo.RangeSet(1, n_customers)  # 1-based indexing for customers
    
    # 4. PARAMETERS
    model.savings_interest_rate = pyo.Param(model.I, initialize={i+1: savings_interest_rates[i] for i in range(n_customers)})
    model.checking_interest_rate = pyo.Param(model.I, initialize={i+1: checking_interest_rates[i] for i in range(n_customers)})
    model.total_funds = pyo.Param(model.I, initialize={i+1: total_funds[i] for i in range(n_customers)})
    model.min_savings_balance = pyo.Param(model.I, initialize={i+1: min_savings_balance[i] for i in range(n_customers)})
    model.min_checking_balance = pyo.Param(model.I, initialize={i+1: min_checking_balance[i] for i in range(n_customers)})
    
    # 5. VARIABLES
    model.savings = pyo.Var(model.I, within=pyo.NonNegativeReals)
    model.checking = pyo.Var(model.I, within=pyo.NonNegativeReals)
    
    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.savings_interest_rate[i] * model.savings[i] + model.checking_interest_rate[i] * model.checking[i] for i in model.I)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.maximize)
    
    # 7. CONSTRAINTS
    
    # Total Funds Constraint
    def total_funds_rule(model, i):
        return model.savings[i] + model.checking[i] <= model.total_funds[i]
    model.total_funds_constraint = pyo.Constraint(model.I, rule=total_funds_rule)
    
    # Minimum Savings Balance Constraint
    def min_savings_rule(model, i):
        return model.savings[i] >= model.min_savings_balance[i]
    model.min_savings_constraint = pyo.Constraint(model.I, rule=min_savings_rule)
    
    # Minimum Checking Balance Constraint
    def min_checking_rule(model, i):
        return model.checking[i] >= model.min_checking_balance[i]
    model.min_checking_constraint = pyo.Constraint(model.I, rule=min_checking_rule)
    
    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')
    
    # Solve the model
    results = solver.solve(model, tee=True)
    
    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective):.2f}")
        
        # Extract variable values
        print("\nVariable values:")
        for i in model.I:
            savings_val = pyo.value(model.savings[i])
            checking_val = pyo.value(model.checking[i])
            print(f"Customer {i}: Savings = {savings_val:.2f}, Checking = {checking_val:.2f}")
        
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")
    
    return model

# Execute the optimization
small_bank_optimization()
```

### Execution Results
**Status**: OPTIMAL
**Optimal Value**: 469.5
**Execution Time**: 1.67 seconds
**Reliability**: medium
**Retry Attempt**: 2
**Analysis**: Pyomo found an optimal solution with a different value, suggesting potential issues with model setup or solver configuration.

## 8. Cross-Solver Analysis and Final Recommendation

### Solver Results Comparison

| Solver | Status | Optimal Value | Execution Time | Decision Variables | Retry Attempt |
|--------|--------|---------------|----------------|-------------------|---------------|
| Gurobipy | OPTIMAL | 458.00 | 0.15s | N/A | 2 |
| Docplex | OPTIMAL | 458.00 | 1.70s | N/A | 2 |
| Pyomo | OPTIMAL | 469.50 | 1.67s | N/A | 2 |

### Solver Consistency Analysis
**Result**: Solvers produced inconsistent results
**Consistent Solvers**: gurobipy, docplex
**Inconsistent Solvers**: pyomo
**Potential Issues**:
- Pyomo might have a different interpretation of constraints or objective function.
- Numerical precision or solver settings could differ between solvers.
- Potential data input errors specific to Pyomo.
**Majority Vote Optimal Value**: 458.0
**Solver Retry Summary**: gurobipy: 2 attempts, docplex: 2 attempts, pyomo: 2 attempts

### Final Recommendation
**Recommended Optimal Value**: 458.0
**Confidence Level**: HIGH
**Preferred Solver(s)**: gurobipy
**Reasoning**: Gurobipy provided a consistent and reliable solution quickly, aligning with DOCplex results.

### Business Interpretation
**Overall Strategy**: The optimal allocation of funds maximizes interest earnings while adhering to constraints.
**Objective Value Meaning**: The optimal objective value represents the maximum total interest earned from all customers.
**Resource Allocation Summary**: Allocate funds to maximize interest while ensuring minimum balances are maintained.
**Implementation Recommendations**: Use Gurobipy for future optimizations, ensure data consistency, and verify model setup across solvers.