# Complete PYOMO implementation - Retry Attempt 4

import pyomo.environ as pyo
from pyomo.opt import SolverFactory

def soccer_optimization():
    """Pyomo model for minimizing yellow cards in soccer team selection"""

    # 1. MODEL CREATION
    model = pyo.ConcreteModel()

    # 2. DATA SETUP
    # Player data
    players = [1, 2, 3, 4, 5]
    positions = ['Goalkeeper', 'Defender', 'Midfielder', 'Forward']
    colleges = ['College A', 'College B', 'College C']
    
    # Player attributes
    player_positions = {1: 'Goalkeeper', 2: 'Defender', 3: 'Midfielder', 4: 'Forward', 5: 'Defender'}
    player_colleges = {1: 'College A', 2: 'College B', 3: 'College C', 4: 'College A', 5: 'College B'}
    yellow_cards = {1: 0, 2: 1, 3: 2, 4: 1, 5: 0}

    # Position requirements
    position_requirements = {'Goalkeeper': 1, 'Defender': 4, 'Midfielder': 4, 'Forward': 2}

    # College limits
    college_limits = {'College A': 3, 'College B': 3, 'College C': 3}

    # 3. SETS
    model.Players = pyo.Set(initialize=players)
    model.Positions = pyo.Set(initialize=positions)
    model.Colleges = pyo.Set(initialize=colleges)

    # 4. PARAMETERS
    model.player_positions = pyo.Param(model.Players, initialize=player_positions)
    model.player_colleges = pyo.Param(model.Players, initialize=player_colleges)
    model.yellow_cards = pyo.Param(model.Players, initialize=yellow_cards)
    model.position_requirements = pyo.Param(model.Positions, initialize=position_requirements)
    model.college_limits = pyo.Param(model.Colleges, initialize=college_limits)

    # 5. VARIABLES
    model.x = pyo.Var(model.Players, within=pyo.Binary)

    # 6. OBJECTIVE FUNCTION
    def obj_rule(model):
        return sum(model.yellow_cards[i] * model.x[i] for i in model.Players)
    model.objective = pyo.Objective(rule=obj_rule, sense=pyo.minimize)

    # 7. CONSTRAINTS

    # Position requirements
    def position_constraint_rule(model, p):
        return sum(model.x[i] for i in model.Players if model.player_positions[i] == p) == model.position_requirements[p]
    model.position_constraints = pyo.Constraint(model.Positions, rule=position_constraint_rule)

    # College diversity
    def college_constraint_rule(model, c):
        return sum(model.x[i] for i in model.Players if model.player_colleges[i] == c) <= model.college_limits[c]
    model.college_constraints = pyo.Constraint(model.Colleges, rule=college_constraint_rule)

    # 8. SOLVING WITH GUROBI
    solver = SolverFactory('gurobi')

    # Solve the model
    results = solver.solve(model, tee=True)

    # 9. RESULT PROCESSING
    if results.solver.termination_condition == pyo.TerminationCondition.optimal:
        print("Optimal solution found!")
        print(f"Optimal value: {pyo.value(model.objective)}")

        # Extract variable values
        print("\nSelected players:")
        for i in model.Players:
            if pyo.value(model.x[i]) > 0.5:  # Binary variable, check if selected
                print(f"Player {i} selected")
    elif results.solver.termination_condition == pyo.TerminationCondition.infeasible:
        print("Problem is infeasible")
    elif results.solver.termination_condition == pyo.TerminationCondition.unbounded:
        print("Problem is unbounded")
    else:
        print(f"Solver terminated with condition: {results.solver.termination_condition}")

    return model

# Run the optimization
soccer_optimization()