# Complete DOCPLEX implementation - Retry Attempt 4

from docplex.mp.model import Model

def train_scheduling_optimization():
    # 1. MODEL & DATA SETUP
    mdl = Model(name="station_weather_optimization")
    
    # Constants and parameters
    base_delay = 1.0  # Example base delay factor
    precipitation_coefficient = 0.05
    wind_speed_coefficient = 0.03
    
    # Example weather data for stations
    precipitation_levels = {101: 10, 102: 5, 103: 8}
    wind_speeds = {101: 15, 102: 10, 103: 12}
    
    # Train travel times data
    train_travel_times = {
        (1, 101): 125.0,
        (2, 102): 135.0,
        (3, 103): 115.0
    }
    
    # Validate data lengths
    station_ids = list(precipitation_levels.keys())
    assert len(station_ids) == len(wind_speeds), "Weather data length mismatch"
    
    # Safe range for indexing
    safe_range = range(len(station_ids))
    
    # 2. VARIABLES
    delay_factors = {s: mdl.continuous_var(name=f"d_{s}", lb=0) for s in station_ids}
    
    # 3. OBJECTIVE FUNCTION
    objective = mdl.sum(train_travel_times[t, s] * delay_factors[s] for t, s in train_travel_times)
    mdl.minimize(objective)
    
    # 4. CONSTRAINTS
    for s in station_ids:
        mdl.add_constraint(
            delay_factors[s] == base_delay + 
            precipitation_coefficient * precipitation_levels[s] + 
            wind_speed_coefficient * wind_speeds[s],
            ctname=f"delay_factor_{s}"
        )
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for s in station_ids:
            value = solution.get_value(delay_factors[s])
            print(f"Delay factor at station {s}: {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

# Run the optimization
train_scheduling_optimization()