# Complete GUROBIPY implementation - Retry Attempt 4

import gurobipy as gp
from gurobipy import GRB

def train_scheduling_optimization():
    """Optimize train scheduling by minimizing total travel time considering weather delays."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("station_weather")
    
    # Data: Adjusted travel times and weather coefficients
    train_travel_times = {
        (1, 101): 125.0,
        (2, 102): 135.0,
        (3, 103): 115.0
    }
    
    base_delay = 1.0
    precipitation_coefficient = 0.05
    wind_speed_coefficient = 0.03
    
    # Weather data (assumed to be available)
    precipitation_levels = {
        101: 10.0,
        102: 5.0,
        103: 8.0
    }
    
    wind_speeds = {
        101: 20.0,
        102: 15.0,
        103: 10.0
    }
    
    # Validate data lengths
    assert len(precipitation_levels) == len(wind_speeds), "Weather data length mismatch"
    
    # 2. VARIABLES
    delay_factors = model.addVars(precipitation_levels.keys(), vtype=GRB.CONTINUOUS, name="d", lb=0)
    
    # 3. OBJECTIVE FUNCTION
    model.setObjective(
        gp.quicksum(train_travel_times[t, s] * delay_factors[s] for t, s in train_travel_times.keys()), 
        GRB.MINIMIZE
    )
    
    # 4. CONSTRAINTS
    model.addConstrs(
        (delay_factors[s] == base_delay + precipitation_coefficient * precipitation_levels[s] + wind_speed_coefficient * wind_speeds[s]
         for s in precipitation_levels.keys()), 
        name="delay_factor_calculation"
    )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for s in precipitation_levels.keys():
            print(f"Delay factor at station {s}: {delay_factors[s].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    else:
        print("Optimization ended with status", model.status)

# Run the optimization
train_scheduling_optimization()