# Complete DOCPLEX implementation

from docplex.mp.model import Model

def university_basketball_optimization():
    """Optimize resource allocation for university basketball teams"""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="university_basketball")
    
    # Data from the problem statement
    all_games_percent = [0.75, 0.6, 0.85]
    minimum_allocation = [50, 60, 70]
    maximum_allocation = [200, 180, 190]
    total_resources_available = 400  # Example total resources available
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(all_games_percent) == len(minimum_allocation) == len(maximum_allocation), "Array length mismatch"
    safe_range = range(min(len(all_games_percent), len(minimum_allocation), len(maximum_allocation)))  # Safe indexing
    
    # 2. VARIABLES
    # Variable dictionaries (most common)
    x = {i: mdl.continuous_var(name=f"x_{i}", lb=0) for i in safe_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the sum of the product of each team's win percentage and the resources allocated to them
    objective = mdl.sum(all_games_percent[i] * x[i] for i in safe_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS - CORRECT SYNTAX PATTERNS
    
    # Total resources constraint
    total_usage = mdl.sum(x[i] for i in safe_range)
    mdl.add_constraint(total_usage <= total_resources_available, ctname="total_resources_limit")
    
    # Minimum and maximum allocation constraints
    for i in safe_range:
        mdl.add_constraint(x[i] >= minimum_allocation[i], ctname=f"min_allocation_{i}")
        mdl.add_constraint(x[i] <= maximum_allocation[i], ctname=f"max_allocation_{i}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_range:
            value = solution.get_value(x[i])
            if value > 1e-6:
                print(f"x[{i}] = {value:.3f}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")
    
    return mdl

# Run the optimization
university_basketball_optimization()