# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_wine_production():
    """Optimize wine production to maximize revenue"""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("wine_optimization")
    
    # Data from the problem context
    prices = [12.0, 18.0, 22.0]
    market_demands = [500, 800, 300]
    total_capacity = 500
    
    # Validate array lengths
    n_wines = len(prices)
    assert len(market_demands) == n_wines, "Array length mismatch"
    
    # 2. VARIABLES
    # Decision variables for the number of cases to produce for each wine
    x = {i: model.addVar(vtype=GRB.CONTINUOUS, name=f"x_{i}", lb=0) 
         for i in range(n_wines)}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize total revenue
    model.setObjective(gp.quicksum(prices[i] * x[i] for i in range(n_wines)), GRB.MAXIMIZE)
    
    # 4. CONSTRAINTS
    
    # Total production capacity constraint
    model.addConstr(gp.quicksum(x[i] for i in range(n_wines)) <= total_capacity, name="total_capacity")
    
    # Market demand constraints
    for i in range(n_wines):
        model.addConstr(x[i] <= market_demands[i], name=f"market_demand_{i}")
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_wines):
            if x[i].x > 1e-6:
                print(f"x[{i}] = {x[i].x:.3f}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_wine_production()