# Complete DOCPLEX implementation

from docplex.mp.model import Model

def optimize_wrestler_assignments():
    """Optimize wrestler assignments to maximize days titles are held."""
    
    # 1. MODEL & DATA SETUP
    mdl = Model(name="wrestler_assignment")
    
    # Wrestler data
    wrestler_ids = [1, 2, 3]
    days_held = [30, 45, 60]
    max_events = [5, 6, 7]
    
    # Event data
    event_ids = [101, 102, 103]
    max_participants = [10, 12, 15]
    
    # CRITICAL: Validate array lengths to prevent IndexError
    assert len(wrestler_ids) == len(days_held) == len(max_events), "Wrestler array length mismatch"
    assert len(event_ids) == len(max_participants), "Event array length mismatch"
    
    # Safe indexing
    safe_wrestler_range = range(len(wrestler_ids))
    safe_event_range = range(len(event_ids))
    
    # 2. VARIABLES
    # Decision variables: x[i][j] = 1 if wrestler i is assigned to event j
    x = {(i, j): mdl.binary_var(name=f"x_{wrestler_ids[i]}_{event_ids[j]}") for i in safe_wrestler_range for j in safe_event_range}
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total number of days titles are held
    objective = mdl.sum(days_held[i] * x[i, j] for i in safe_wrestler_range for j in safe_event_range)
    mdl.maximize(objective)
    
    # 4. CONSTRAINTS
    
    # Wrestler Participation Constraints
    for i in safe_wrestler_range:
        mdl.add_constraint(mdl.sum(x[i, j] for j in safe_event_range) <= max_events[i], ctname=f"max_events_{wrestler_ids[i]}")
    
    # Event Capacity Constraints
    for j in safe_event_range:
        mdl.add_constraint(mdl.sum(x[i, j] for i in safe_wrestler_range) <= max_participants[j], ctname=f"max_participants_{event_ids[j]}")
    
    # 5. SOLVING & RESULTS
    solution = mdl.solve()
    
    if solution:
        print(f"Optimal value: {solution.objective_value}")
        for i in safe_wrestler_range:
            for j in safe_event_range:
                value = solution.get_value(x[i, j])
                if value > 1e-6:
                    print(f"Wrestler {wrestler_ids[i]} assigned to Event {event_ids[j]}: {value}")
    else:
        print("No solution found")
        print(f"Status: {mdl.solve_details.status}")

    return mdl

# Run the optimization
optimize_wrestler_assignments()