# Complete GUROBIPY implementation

import gurobipy as gp
from gurobipy import GRB

def optimize_wrestler_assignments():
    """Optimize wrestler assignments to maximize days titles are held."""
    
    # 1. MODEL & DATA SETUP
    model = gp.Model("wrestler_assignment")
    
    # Wrestler data
    wrestlers = [
        {"WrestlerID": 1, "Days_held": 30, "MaxEvents": 5},
        {"WrestlerID": 2, "Days_held": 45, "MaxEvents": 6},
        {"WrestlerID": 3, "Days_held": 60, "MaxEvents": 7}
    ]
    
    # Event constraints
    events = [
        {"EventID": 101, "MaxParticipants": 10},
        {"EventID": 102, "MaxParticipants": 12},
        {"EventID": 103, "MaxParticipants": 15}
    ]
    
    # Validate data lengths
    n_wrestlers = len(wrestlers)
    n_events = len(events)
    assert n_wrestlers > 0 and n_events > 0, "No wrestlers or events available"
    
    # 2. VARIABLES
    # Decision variables: x[i, j] = 1 if wrestler i is assigned to event j
    x = model.addVars(n_wrestlers, n_events, vtype=GRB.BINARY, name="x")
    
    # 3. OBJECTIVE FUNCTION
    # Maximize the total number of days titles are held
    model.setObjective(
        gp.quicksum(wrestlers[i]["Days_held"] * x[i, j] for i in range(n_wrestlers) for j in range(n_events)),
        GRB.MAXIMIZE
    )
    
    # 4. CONSTRAINTS
    # Wrestler participation constraints
    for i in range(n_wrestlers):
        model.addConstr(
            gp.quicksum(x[i, j] for j in range(n_events)) <= wrestlers[i]["MaxEvents"],
            name=f"wrestler_max_events_{i}"
        )
    
    # Event capacity constraints
    for j in range(n_events):
        model.addConstr(
            gp.quicksum(x[i, j] for i in range(n_wrestlers)) <= events[j]["MaxParticipants"],
            name=f"event_max_participants_{j}"
        )
    
    # 5. SOLVING & RESULTS
    model.optimize()
    
    if model.status == GRB.OPTIMAL:
        print(f"Optimal value: {model.objVal}")
        for i in range(n_wrestlers):
            for j in range(n_events):
                if x[i, j].x > 1e-6:
                    print(f"Wrestler {wrestlers[i]['WrestlerID']} assigned to Event {events[j]['EventID']}")
    elif model.status == GRB.INFEASIBLE:
        print("Problem is infeasible")
    elif model.status == GRB.UNBOUNDED:
        print("Problem is unbounded")
    
    return model

# Run the optimization
optimize_wrestler_assignments()