import random

import numpy as np
import torch
from torch import nn
import torch.nn.functional as F
import tqdm
import argparse
import json
from data_builder import load_data
from model import load_tokenizer, load_model
from metrics import get_roc_metrics, get_precision_recall_metrics
from nuisance_func import BSplineW, ShiftLearner
from utils import load_training_data, separated_string
import json

def get_samples(logits, labels):
    assert logits.shape[0] == 1
    assert labels.shape[0] == 1
    nsamples = 10000
    lprobs = torch.log_softmax(logits, dim=-1)
    distrib = torch.distributions.categorical.Categorical(logits=lprobs)
    samples = distrib.sample([nsamples]).permute([1, 2, 0])
    return samples

def get_likelihood(logits, labels):
    assert logits.shape[0] == 1
    assert labels.shape[0] == 1
    labels = labels.unsqueeze(-1) if labels.ndim == logits.ndim - 1 else labels
    lprobs = torch.log_softmax(logits, dim=-1)
    log_likelihood = lprobs.gather(dim=-1, index=labels)
    return log_likelihood.mean(dim=1)

def get_token_likelihood(logits, labels):
    assert logits.shape[0] == 1
    if labels is not None:
        assert labels.shape[0] == 1
        labels = labels.unsqueeze(-1) if labels.ndim == logits.ndim - 1 else labels
    lprobs = torch.log_softmax(logits, dim=-1)
    if labels is not None:
        log_likelihood = lprobs.gather(dim=-1, index=labels)
    else:
        log_likelihood = lprobs
    return log_likelihood

def get_martingale_stat(logits_ref, logits_score, labels, burn_in_num, w_func, shift_value=None):
    assert logits_ref.shape[0] == 1
    assert logits_score.shape[0] == 1
    assert labels.shape[0] == 1
    text_length = labels.size(-1)
    assert burn_in_num < text_length
    
    if logits_ref.size(-1) != logits_score.size(-1):
        # print(f"WARNING: vocabulary size mismatch {logits_ref.size(-1)} vs {logits_score.size(-1)}.")
        vocab_size = min(logits_ref.size(-1), logits_score.size(-1))
        logits_ref = logits_ref[:, :, :vocab_size]
        logits_score = logits_score[:, :, :vocab_size]

    log_likelihood_x = w_func(get_token_likelihood(logits_score, labels))
    log_likelihood_x_full = w_func(get_token_likelihood(logits_score, None))
    p_x_full = torch.softmax(logits_ref, dim=-1)
    mu_x = (log_likelihood_x_full * p_x_full).sum(dim=-1)
    sigma_x = (torch.square(log_likelihood_x_full - mu_x.unsqueeze(-1)) * p_x_full).sum(dim=-1).sqrt()
    discrepancy = (log_likelihood_x.squeeze(-1) - mu_x) / sigma_x
    discrepancy = discrepancy[0, burn_in_num:]
    stats = discrepancy.sum(dim=-1) / torch.sqrt(torch.tensor(text_length - burn_in_num))
    return stats.item()

def get_classification_stat(logits_ref, logits_score, labels, burn_in_num, w_func, shift_value=None):
    assert logits_ref.shape[0] == 1
    assert logits_score.shape[0] == 1
    assert labels.shape[0] == 1
    if logits_ref.size(-1) != logits_score.size(-1):
        # print(f"WARNING: vocabulary size mismatch {logits_ref.size(-1)} vs {logits_score.size(-1)}.")
        vocab_size = min(logits_ref.size(-1), logits_score.size(-1))
        logits_ref = logits_ref[:, :, :vocab_size]
        logits_score = logits_score[:, :, :vocab_size]

    labels = labels.unsqueeze(-1) if labels.ndim == logits_score.ndim - 1 else labels
    lprobs_score = w_func(torch.log_softmax(logits_score, dim=-1))
    probs_ref = torch.softmax(logits_ref, dim=-1)
    # max_idx = probs_ref.argmax(dim=-1, keepdim=True)        
    # probs_ref = torch.zeros_like(probs_ref).scatter_(       
    #     dim=-1,
    #     index=max_idx,
    #     src=torch.ones_like(max_idx, dtype=probs_ref.dtype)  
    # )
    log_likelihood = lprobs_score.gather(dim=-1, index=labels).squeeze(-1)
    mean_ref = (probs_ref * lprobs_score).sum(dim=-1)
    var_ref = (probs_ref * torch.square(lprobs_score)).sum(dim=-1) - torch.square(mean_ref)
    # var_ref = torch.ones_like(mean_ref)
    log_likelihood = log_likelihood[0, burn_in_num:]
    mean_ref = mean_ref[0, burn_in_num:]
    var_ref = var_ref[0, burn_in_num:]
    L = torch.tensor(var_ref.shape[0])
    stat = (log_likelihood.sum(dim=-1) - mean_ref.sum(dim=-1) - shift_value * L) / var_ref.sum(dim=-1).sqrt()
    stat = stat.mean()
    return stat.item()

def experiment(args):
    # load model
    scoring_tokenizer = load_tokenizer(args.scoring_model_name, args.cache_dir)
    scoring_model = load_model(args.scoring_model_name, args.device, args.cache_dir)
    scoring_model.eval()
    if args.sampling_model_name != args.scoring_model_name:
        sampling_tokenizer = load_tokenizer(args.sampling_model_name, args.cache_dir)
        sampling_model = load_model(args.sampling_model_name, args.device, args.cache_dir)
        sampling_model.eval()
    # load data
    data = load_data(args.dataset_file)
    n_samples = len(data["sampled"])
    # evaluate criterion
    if args.discrepancy == 'martingale':
        name = "martingale"
        criterion_fn = get_martingale_stat
    elif args.discrepancy == 'classification':
        name = "classification"
        criterion_fn = get_classification_stat

    # w function
    if args.w_func == 'identity':
        w_func = nn.Identity()
        beta = None
    elif args.w_func == 'bspline':
        bspline_args = args.config
        ## load training data
        print(f"Datasets for learning BSpline: {args.train_dataset}")
        w_func = BSplineW(bspline_args)
        train_data = load_training_data(args.train_dataset)
        w_func.fit(train_data, scoring_tokenizer, scoring_model, args)
        beta = w_func.beta_hat.detach().cpu().tolist()
    
    if args.debias:
        print(f"Datasets for learning bias: {args.train_dataset}")
        train_data = load_training_data(args.train_dataset)
        shift_learner = ShiftLearner()
        shift_learner.fit(train_data, scoring_tokenizer, scoring_model, w_func, args)
        shift_value = shift_learner.c_hat
    else:
        shift_value = torch.zeros(1).to(args.device)
    
    random.seed(args.seed)
    torch.manual_seed(args.seed)
    np.random.seed(args.seed)
    results = []
    for idx in tqdm.tqdm(range(n_samples), desc=f"Computing {name} criterion"):
        original_text = data["original"][idx]
        sampled_text = data["sampled"][idx]
        # original text
        tokenized = scoring_tokenizer(original_text, return_tensors="pt", padding=True, return_token_type_ids=False).to(args.device)
        labels = tokenized.input_ids[:, 1:]
        if args.burn_in < 1.0:
            burn_in_num = int(args.burn_in * labels.size(-1))
        else:
            burn_in_num = int(args.burn_in)
        with torch.no_grad():
            logits_score = scoring_model(**tokenized).logits[:, :-1]
            if args.sampling_model_name == args.scoring_model_name:
                logits_ref = logits_score
            else:
                tokenized = sampling_tokenizer(original_text, return_tensors="pt", padding=True, return_token_type_ids=False).to(args.device)
                assert torch.all(tokenized.input_ids[:, 1:] == labels), "Tokenizer is mismatch."
                logits_ref = sampling_model(**tokenized).logits[:, :-1]
            original_crit = criterion_fn(logits_ref, logits_score, labels, burn_in_num, w_func, shift_value)
        # sampled text
        tokenized = scoring_tokenizer(sampled_text, return_tensors="pt", padding=True, return_token_type_ids=False).to(args.device)
        labels = tokenized.input_ids[:, 1:]
        if args.burn_in < 1.0:
            burn_in_num = int(args.burn_in * labels.size(-1))
        else:
            burn_in_num = int(args.burn_in)
        with torch.no_grad():
            logits_score = scoring_model(**tokenized).logits[:, :-1]
            if args.sampling_model_name == args.scoring_model_name:
                logits_ref = logits_score
            else:
                tokenized = sampling_tokenizer(sampled_text, return_tensors="pt", padding=True, return_token_type_ids=False).to(args.device)
                assert torch.all(tokenized.input_ids[:, 1:] == labels), "Tokenizer is mismatch."
                logits_ref = sampling_model(**tokenized).logits[:, :-1]
            sampled_crit = criterion_fn(logits_ref, logits_score, labels, burn_in_num, w_func, shift_value)
        # result
        results.append({"original": original_text,
                        "original_crit": original_crit,
                        "sampled": sampled_text,
                        "sampled_crit": sampled_crit})
        # print(f"Real mean: {original_crit:.2f}, Samples mean/std: {sampled_crit:.2f}")

    # compute prediction scores for real/sampled passages
    predictions = {'real': [x["original_crit"] for x in results],
                   'samples': [x["sampled_crit"] for x in results]}
    print(f"Real mean/std: {np.mean(predictions['real']):.2f}/{np.std(predictions['real']):.2f}, Samples mean/std: {np.mean(predictions['samples']):.2f}/{np.std(predictions['samples']):.2f}")
    fpr, tpr, roc_auc = get_roc_metrics(predictions['real'], predictions['samples'])
    p, r, pr_auc = get_precision_recall_metrics(predictions['real'], predictions['samples'])
    print(f"Criterion {name}_threshold ROC AUC: {roc_auc:.4f}, PR AUC: {pr_auc:.4f}")
    # results
    if args.output_info:
        if args.debias:
            results_file = f'{args.output_file}.{name}.{args.w_func}.debias.json'
        else:
            results_file = f'{args.output_file}.{name}.{args.w_func}.json'
    else:
        if args.debias:
            results_file = f'{args.output_file}.{name}.debias.json'
        else:
            results_file = f'{args.output_file}.{name}.json'
    results = { 'name': f'{name}_threshold',
                'info': {'n_samples': n_samples},
                'predictions': predictions,
                'raw_results': results,
                'metrics': {'roc_auc': roc_auc, 'fpr': fpr, 'tpr': tpr},
                'pr_metrics': {'pr_auc': pr_auc, 'precision': p, 'recall': r},
                'loss': 1 - pr_auc, 
                'beta': beta, 
                'bias': shift_value.detach().cpu().tolist(),
                }
    with open(results_file, 'w') as fout:
        json.dump(results, fout)
        print(f'Results written into {results_file}')

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--output_file', type=str, default="./exp_main/results/xsum_opt-2.7b")  # output file prefix
    parser.add_argument('--dataset', type=str, default="xsum")
    parser.add_argument('--dataset_file', type=str, default="./exp_main/data/xsum_opt-2.7b")
    parser.add_argument('--train_dataset', type=separated_string, default="./exp_main/data/squad_opt-2.7b&./exp_main/data/writing_opt-2.7b")
    parser.add_argument('--sampling_model_name', type=str, default="opt-2.7b")
    parser.add_argument('--scoring_model_name', type=str, default="opt-2.7b")
    parser.add_argument('--discrepancy', type=str, default='classification', choices=['classification', 'martingale'])
    parser.add_argument('--burn_in', type=float, default=0.0)
    parser.add_argument('--w_func', type=str, default='bspline', choices=['identity', 'bspline'])
    parser.add_argument("--config", type=json.loads, default='{"start": -32, "end": 0, "n_bases": 16, "spline_order": 2}', help='A JSON dict')
    parser.add_argument('--debias', action='store_true')
    parser.add_argument('--output_info', action='store_true')
    parser.add_argument('--seed', type=int, default=0)
    parser.add_argument('--device', type=str, default="cuda")
    parser.add_argument('--cache_dir', type=str, default="../cache")
    args = parser.parse_args()

    experiment(args)
