# SPDX-FileCopyrightText: Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# SPDX-License-Identifier: BSD-3-Clause
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice, this
# list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Neither the name of the copyright holder nor the names of its
# contributors may be used to endorse or promote products derived from
# this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# Copyright (c) 2021 ETH Zurich, Nikita Rudin

import numpy as np

import code
import torch
import torch.nn as nn
from torch.distributions import Normal
from torch.nn.modules import rnn
from torch.nn.modules.activation import ReLU

from rsl_rl.modules.depth_backbone import DepthOnlyFCBackbone58x87


class StateHistoryEncoder(nn.Module):
    def __init__(self, activation_fn, input_size, tsteps, output_size, tanh_encoder_output=False):
        # self.device = device
        super(StateHistoryEncoder, self).__init__()
        self.activation_fn = activation_fn
        self.tsteps = tsteps

        channel_size = 10
        # last_activation = nn.ELU()

        self.encoder = nn.Sequential(
                nn.Linear(input_size, 3 * channel_size), self.activation_fn,
                )

        if tsteps == 50:
            self.conv_layers = nn.Sequential(
                    nn.Conv1d(in_channels = 3 * channel_size, out_channels = 2 * channel_size, kernel_size = 8, stride = 4), self.activation_fn,
                    nn.Conv1d(in_channels = 2 * channel_size, out_channels = channel_size, kernel_size = 5, stride = 1), self.activation_fn,
                    nn.Conv1d(in_channels = channel_size, out_channels = channel_size, kernel_size = 5, stride = 1), self.activation_fn, nn.Flatten())
        elif tsteps == 10:
            self.conv_layers = nn.Sequential(
                nn.Conv1d(in_channels = 3 * channel_size, out_channels = 2 * channel_size, kernel_size = 4, stride = 2), self.activation_fn,
                nn.Conv1d(in_channels = 2 * channel_size, out_channels = channel_size, kernel_size = 2, stride = 1), self.activation_fn,
                nn.Flatten())
        elif tsteps == 20:
            self.conv_layers = nn.Sequential(
                nn.Conv1d(in_channels = 3 * channel_size, out_channels = 2 * channel_size, kernel_size = 6, stride = 2), self.activation_fn,
                nn.Conv1d(in_channels = 2 * channel_size, out_channels = channel_size, kernel_size = 4, stride = 2), self.activation_fn,
                nn.Flatten())

        # This has changed
        elif tsteps == 0:
            self.conv_layers = nn.Sequential(
                nn.Conv1d(in_channels = 3 * channel_size, out_channels = 2 * channel_size, kernel_size = 4, stride = 2), self.activation_fn,
                nn.Conv1d(in_channels = 2 * channel_size, out_channels = channel_size, kernel_size = 2, stride = 1), self.activation_fn,
                nn.Flatten())
        else:
            raise(ValueError("tsteps must be 10, 20 or 50"))

        self.linear_output = nn.Sequential(
                nn.Linear(channel_size * 3, output_size), self.activation_fn
                )

    def forward(self, obs):
        # nd * T * n_proprio
        nd = obs.shape[0]
        T = self.tsteps
        # print("obs device", obs.device)
        # print("encoder device", next(self.encoder.parameters()).device)
        projection = self.encoder(obs.reshape([nd * T, -1])) # do projection for n_proprio -> 32
        output = self.conv_layers(projection.reshape([nd, T, -1]).permute((0, 2, 1)))
        output = self.linear_output(output)
        return output

class Actor(nn.Module):
    def __init__(self, num_prop, 
                 num_demo,
                 text_feat_input_dim,
                 text_feat_output_dim,
                 feat_hist_len,
                 num_scan, 
                 num_actions, 
                 scan_encoder_dims,
                 actor_hidden_dims, 
                 depth_encoder_dims, 
                 n_decoder_out, 
                 num_priv_explicit, 
                 num_hist, activation, 
                 tanh_encoder_output=False) -> None:
        super().__init__()
        # motion_features -> prop -> demo -> scan -> priv_explicit -> priv_latent -> hist
        self.num_prop = num_prop
        self.num_demo = num_demo
        self.num_scan = num_scan
        self.num_hist = num_hist
        self.num_actions = num_actions
        self.n_decoder_out = n_decoder_out
        self.num_priv_explicit = num_priv_explicit
        self.if_scan_encode = scan_encoder_dims is not None and num_scan > 0
        self.text_feat_input_dim = text_feat_input_dim
        self.text_feat_output_dim = text_feat_output_dim
        self.num_feat_prop_hist = feat_hist_len

        self.n_depth = depth_encoder_dims
        if self.n_depth == 0:
            depth_feature = 0
        else:
            depth_feature = 32
        # For CNN depth 
        self.depth_encoder = DepthOnlyFCBackbone58x87(depth_feature)
        
        text_feat_encoder = []
        text_feat_encoder.append(nn.Linear(self.num_prop, 128))
        text_feat_encoder.append(activation)
        text_feat_encoder.append(nn.Linear(128, text_feat_output_dim))
        text_feat_encoder.append(activation)
        self.text_feat_encoder = nn.Sequential(*text_feat_encoder)
        
        text_feat_merger = []
        text_feat_merger.append(nn.Linear(text_feat_output_dim*feat_hist_len, text_feat_output_dim))
        text_feat_merger.append(activation)
        self.text_feat_merger = nn.Sequential(*text_feat_merger)


        # Ours
        actor_layers = []
        actor_layers.append(nn.Linear(text_feat_output_dim + num_prop + num_demo + depth_feature + n_decoder_out, actor_hidden_dims[0]))
        # import pdb
        # pdb.set_trace()
        # print("The demo obs is", num_demo)

        actor_layers.append(activation)
        for l in range(len(actor_hidden_dims)):
            if l == len(actor_hidden_dims) - 1:
                actor_layers.append(nn.Linear(actor_hidden_dims[l], num_actions))
            else:
                actor_layers.append(nn.Linear(actor_hidden_dims[l], actor_hidden_dims[l + 1]))
                actor_layers.append(activation)
        if tanh_encoder_output:
            actor_layers.append(nn.Tanh())
        self.actor_backbone = nn.Sequential(*actor_layers)


    # # For VAE
    # def forward(self, obs_all, hist_encoding: bool, eval=False, scandots_latent=None):
    #     text_feat = obs_all[:, :self.text_feat_input_dim].reshape(-1, self.num_prop)
    #     text_feat_latent = self.text_feat_encoder(text_feat).view(obs_all.shape[0], -1)
    #     text_feat_latent = self.text_feat_merger(text_feat_latent)

    #     obs = obs_all[:, self.text_feat_input_dim:]
    #     obs_prop = obs[:, : self.num_prop]

    #     obs_priv_explicit = obs[:, self.num_prop + self.n_depth : self.num_prop + self.n_depth + self.n_decoder_out]

    #     if self.n_depth == 0:
    #         backbone_input = torch.cat([text_feat_latent, obs_prop, obs_priv_explicit], dim=1)
    #     else:
    #         obs_depth = obs[:, self.num_prop : self.num_prop + self.n_depth].reshape(-1, 58, 87)   
    #         depth_feature = self.depth_encoder(obs_depth)
    #         backbone_input = torch.cat([text_feat_latent, obs_prop, depth_feature, obs_priv_explicit], dim=1)

    #     backbone_output = self.actor_backbone(backbone_input)
    #     return backbone_output

    # For VAE
    def forward(self, obs_all, hist_encoding: bool, eval=False, scandots_latent=None):
        text_feat = obs_all[:, :self.text_feat_input_dim].reshape(-1, self.num_prop)
        text_feat_latent = self.text_feat_encoder(text_feat).view(obs_all.shape[0], -1)
        text_feat_latent = self.text_feat_merger(text_feat_latent)

        obs = obs_all[:, self.text_feat_input_dim:]
        obs_prop = obs[:, : self.num_prop]
        obs_demo = obs[:, self.num_prop : self.num_prop + self.num_demo]

        obs_priv_explicit = obs[:, - self.n_decoder_out : ]

        if self.n_depth == 0:
            backbone_input = torch.cat([text_feat_latent, obs_prop, obs_demo, obs_priv_explicit], dim=1)
        else:
            obs_depth = obs[:, self.num_prop : self.num_prop + self.n_depth].reshape(-1, 58, 87)   
            depth_feature = self.depth_encoder(obs_depth)
            backbone_input = torch.cat([text_feat_latent, obs_prop, obs_demo, depth_feature, obs_priv_explicit], dim=1)

        backbone_output = self.actor_backbone(backbone_input)
        return backbone_output


    def infer_priv_latent(self, obs):
        priv = obs[:, self.text_feat_input_dim + self.num_prop + self.num_demo + self.num_scan + self.num_priv_explicit: self.text_feat_input_dim + self.num_prop + self.num_demo + self.num_scan + self.num_priv_explicit + self.num_priv_latent]
        return self.priv_encoder(priv)
    
    def infer_hist_latent(self, obs):
        hist = obs[:, -self.num_hist*self.num_prop:]
        return self.history_encoder(hist.view(-1, self.num_hist, self.num_prop))
    
    def infer_scandots_latent(self, obs):
        scan = obs[:, self.num_prop + self.num_demo:self.num_prop + self.num_demo + self.num_scan]
        return self.scan_encoder(scan)


# class ActorDistill(nn.Module):
#     def __init__(self, actor: Actor, student_params):
#         super().__init__()
#         self._orig_actor = actor
        
#         for param in self._orig_actor.parameters():
#             param.requires_grad = False
        
#         activation = get_activation(student_params['activation'])
        
#         num_actions = self._orig_actor.num_actions
#         num_prop = self._orig_actor.num_prop
#         num_priv_explicit = self._orig_actor.num_priv_explicit
#         priv_encoder_output_dim = self._orig_actor.priv_encoder[-2].out_features
#         text_feat_output_dim = self._orig_actor.text_feat_output_dim
#         self.num_demo = student_params['num_demo']
#         actor_hidden_dims = student_params['student_actor_hidden_dims']

#         actor_layers = []
#         actor_layers.append(nn.Linear(num_prop + self.num_demo + text_feat_output_dim + 
#                                       self._orig_actor.scan_encoder_output_dim+
#                                       num_priv_explicit+
#                                       priv_encoder_output_dim, 
#                                       actor_hidden_dims[0]))
#         actor_layers.append(activation)
#         for l in range(len(actor_hidden_dims)):
#             if l == len(actor_hidden_dims) - 1:
#                 actor_layers.append(nn.Linear(actor_hidden_dims[l], num_actions))
#             else:
#                 actor_layers.append(nn.Linear(actor_hidden_dims[l], actor_hidden_dims[l + 1]))
#                 actor_layers.append(activation)
#         self.student_actor_backbone = nn.Sequential(*actor_layers)

#     def act_teacher(self, obs_all, hist_encoding: bool, eval=False, scandots_latent=None):
#         with torch.no_grad():
#             return self._orig_actor(obs_all, hist_encoding, eval, scandots_latent)

#     def forward(self, obs_teacher, decoder_demo_obs, hist_encoding: bool, eval=False, scandots_latent=None):
#         text_feat = obs_teacher[:, :self._orig_actor.text_feat_input_dim].reshape(-1, 64)
#         text_feat_latent = self._orig_actor.text_feat_encoder(text_feat).view(obs_teacher.shape[0], -1)
#         text_feat_latent = self._orig_actor.text_feat_merger(text_feat_latent)

#         obs = obs_teacher[:, self._orig_actor.text_feat_input_dim:]
#         obs_prop = obs[:, :self._orig_actor.num_prop]
#         obs_demo = decoder_demo_obs
#         # obs_priv_explicit = obs[:, self._orig_actor.num_prop + self._orig_actor.num_demo + self._orig_actor.num_scan:self._orig_actor.num_prop + self._orig_actor.num_demo + self._orig_actor.num_scan + self._orig_actor.num_priv_explicit]
#         # if hist_encoding:
#         #     latent = self._orig_actor.infer_hist_latent(obs)
#         # else:
#         #     latent = self._orig_actor.infer_priv_latent(obs)
#         # backbone_input = torch.cat([text_feat_latent, obs_prop, obs_demo, obs_priv_explicit, latent], dim=1)
#         zeros_expand = torch.zeros(len(obs[0]), self._orig_actor.num_priv_explicit )
#         backbone_input = torch.cat([text_feat_latent, obs_prop, obs_demo], dim=1)
#         backbone_output = self.student_actor_backbone(backbone_input)
#         return backbone_output
    

class ActorCriticMimic(nn.Module):
    is_recurrent = False
    def __init__(self,  num_prop,
                        num_demo,
                        num_scan,
                        num_critic_obs,
                        n_decoder_out, 
                        num_priv_explicit,
                        num_hist,
                        num_actions,
                        scan_encoder_dims=[256, 256, 256],
                        actor_hidden_dims=[256, 256, 256],
                        critic_hidden_dims=[256, 256, 256],
                        activation='elu',
                        init_noise_std=1.0,
                        **kwargs):
        if kwargs:
            print("ActorCritic.__init__ got unexpected arguments, which will be ignored: " + str([key for key in kwargs.keys()]))
        super(ActorCriticMimic, self).__init__()

        

        self.kwargs = kwargs
        # priv_encoder_dims= kwargs['priv_encoder_dims']
        depth_encoder_dims = kwargs['n_depth']
        activation = get_activation(activation)
        
        # self.actor = Actor(num_prop, num_demo, num_scan, num_actions, scan_encoder_dims, actor_hidden_dims, priv_encoder_dims, num_priv_latent, num_priv_explicit, num_hist, activation, tanh_encoder_output=kwargs['tanh_encoder_output'])
        self.actor = Actor(num_prop, num_demo, kwargs["text_feat_input_dim"], kwargs["text_feat_output_dim"], kwargs["feat_hist_len"], num_scan, num_actions, scan_encoder_dims, actor_hidden_dims, depth_encoder_dims, n_decoder_out, num_priv_explicit, num_hist, activation, tanh_encoder_output=kwargs['tanh_encoder_output'])
        

        # Value function
        # print('num_critic_obs', num_critic_obs)
        critic_layers = []
        critic_layers.append(nn.Linear(num_critic_obs, critic_hidden_dims[0]))
        critic_layers.append(activation)
        for l in range(len(critic_hidden_dims)):
            if l == len(critic_hidden_dims) - 1:
                critic_layers.append(nn.Linear(critic_hidden_dims[l], 1))
            else:
                critic_layers.append(nn.Linear(critic_hidden_dims[l], critic_hidden_dims[l + 1]))
                critic_layers.append(activation)
        self.critic = nn.Sequential(*critic_layers)

        # Action noise
        self.std = nn.Parameter(init_noise_std * torch.ones(num_actions))
        self.distribution = None
        # disable args validation for speedup
        Normal.set_default_validate_args = False
        
        # seems that we get better performance without init
        # self.init_memory_weights(self.memory_a, 0.001, 0.)
        # self.init_memory_weights(self.memory_c, 0.001, 0.)
    
    @staticmethod
    # not used at the moment
    def init_weights(sequential, scales):
        [torch.nn.init.orthogonal_(module.weight, gain=scales[idx]) for idx, module in
         enumerate(mod for mod in sequential if isinstance(mod, nn.Linear))]

    def reset(self, dones=None):
        pass

    def forward(self):
        raise NotImplementedError
    
    @property
    def action_mean(self):
        return self.distribution.mean

    @property
    def action_std(self):
        return self.distribution.stddev
    
    @property
    def entropy(self):
        return self.distribution.entropy().sum(dim=-1)

    def update_distribution(self, observations, hist_encoding):
        mean = self.actor(observations, hist_encoding)
        self.distribution = Normal(mean, mean*0. + self.std)

    def act(self, observations, hist_encoding=False, **kwargs):
        self.update_distribution(observations, hist_encoding)
        return self.distribution.sample()
    
    def get_actions_log_prob(self, actions):
        return self.distribution.log_prob(actions).sum(dim=-1)

    def act_inference(self, observations, hist_encoding=False, eval=False, scandots_latent=None, **kwargs):
        if not eval:
            actions_mean = self.actor(observations, hist_encoding, eval, scandots_latent)
            return actions_mean
        else:
            actions_mean, latent_hist, latent_priv = self.actor(observations, hist_encoding, eval=True)
            return actions_mean, latent_hist, latent_priv

    def evaluate(self, critic_observations, **kwargs):
        value = self.critic(critic_observations)
        return value
    
    def reset_std(self, std, num_actions, device):
        new_std = std * torch.ones(num_actions, device=device)
        self.std.data = new_std.data

def get_activation(act_name):
    if act_name == "elu":
        return nn.ELU()
    elif act_name == "selu":
        return nn.SELU()
    elif act_name == "relu":
        return nn.ReLU()
    elif act_name == "crelu":
        return nn.ReLU()
    elif act_name == "lrelu":
        return nn.LeakyReLU()
    elif act_name == "tanh":
        return nn.Tanh()
    elif act_name == "sigmoid":
        return nn.Sigmoid()
    else:
        print("invalid activation function!")
        return None
