# Copyright (c) OpenMMLab. All rights reserved.
import mmcv
import numpy as np
import warnings
from mmcv import is_tuple_of
from mmcv.utils import build_from_cfg

from mmdet3d.core import VoxelGenerator
from mmdet3d.core.bbox import (CameraInstance3DBoxes, DepthInstance3DBoxes,
                               LiDARInstance3DBoxes, box_np_ops)
from mmdet.datasets.builder import PIPELINES
from mmdet.datasets.pipelines import RandomFlip
from mmdet3d.datasets.builder import OBJECTSAMPLERS
# from mmdet3d.datasets.pipelines.data_augment_utils import noise_per_object_v3_
from mmdet3d.datasets.pipelines.utils import noise_per_object_v3_
from mmdet.datasets.pipelines import LoadAnnotations, LoadImageFromFile

# @PIPELINES.register_module()
# class PadMultiViewImage(object):
#     """Pad the multi-view image.
#     There are two padding modes: (1) pad to a fixed size and (2) pad to the
#     minimum size that is divisible by some number.
#     Added keys are "pad_shape", "pad_fixed_size", "pad_size_divisor",
#     Args:
#         size (tuple, optional): Fixed padding size.
#         size_divisor (int, optional): The divisor of padded size.
#         pad_val (float, optional): Padding value, 0 by default.
#     """

#     def __init__(self, size=None, size_divisor=None, pad_val=0):
#         self.size = size
#         self.size_divisor = size_divisor
#         self.pad_val = pad_val
#         # only one of size and size_divisor should be valid
#         assert size is not None or size_divisor is not None
#         assert size is None or size_divisor is None

#     def _pad_img(self, results):
#         """Pad images according to ``self.size``."""
#         if self.size is not None:
#             padded_img = [mmcv.impad(
#                 img, shape=self.size, pad_val=self.pad_val) for img in results['img']]
#         elif self.size_divisor is not None:
#             padded_img = [mmcv.impad_to_multiple(
#                 img, self.size_divisor, pad_val=self.pad_val) for img in results['img']]
#         results['img'] = padded_img
#         results['img_shape'] = [img.shape for img in padded_img]
#         results['pad_shape'] = [img.shape for img in padded_img]
#         results['pad_fixed_size'] = self.size
#         results['pad_size_divisor'] = self.size_divisor

#     def __call__(self, results):
#         """Call function to pad images, masks, semantic segmentation maps.
#         Args:
#             results (dict): Result dict from loading pipeline.
#         Returns:
#             dict: Updated result dict.
#         """
#         self._pad_img(results)
#         return results

#     def __repr__(self):
#         repr_str = self.__class__.__name__
#         repr_str += f'(size={self.size}, '
#         repr_str += f'size_divisor={self.size_divisor}, '
#         repr_str += f'pad_val={self.pad_val})'
#         return repr_str


# @PIPELINES.register_module()
# class NormalizeMultiviewImage(object):
#     """Normalize the image.
#     Added key is "img_norm_cfg".
#     Args:
#         mean (sequence): Mean values of 3 channels.
#         std (sequence): Std values of 3 channels.
#         to_rgb (bool): Whether to convert the image from BGR to RGB,
#             default is true.
#     """

#     def __init__(self, mean, std, to_rgb=True):
#         self.mean = np.array(mean, dtype=np.float32)
#         self.std = np.array(std, dtype=np.float32)
#         self.to_rgb = to_rgb

#     def __call__(self, results):
#         """Call function to normalize images.
#         Args:
#             results (dict): Result dict from loading pipeline.
#         Returns:
#             dict: Normalized results, 'img_norm_cfg' key is added into
#                 result dict.
#         """
#         results['img'] = [mmcv.imnormalize(
#             img, self.mean, self.std, self.to_rgb) for img in results['img']]
#         results['img_norm_cfg'] = dict(
#             mean=self.mean, std=self.std, to_rgb=self.to_rgb)
#         return results

#     def __repr__(self):
#         repr_str = self.__class__.__name__
#         repr_str += f'(mean={self.mean}, std={self.std}, to_rgb={self.to_rgb})'
#         return repr_str


@PIPELINES.register_module()
class ScaleImageMultiViewImage(object):
    """scale the image
    Args:
        scales
    """

    def __init__(self, scales=(800, 448)):
        self.scales = np.array(scales)
        self.scales[0] = self.scales[0] + self.scales[1]
        self.scales[1] = self.scales[0] - self.scales[1]
        self.scales[0] = self.scales[0] - self.scales[1]

    def __call__(self, results):
        """Call function to pad images, masks, semantic segmentation maps.
        Args:
            results (dict): Result dict from loading pipeline.
        Returns:
            dict: Updated result dict.
        """
        img_shape = results['img_shape']
        rand_scale = self.scales / np.array(img_shape[:2])
        y_size = int(img_shape[0] * rand_scale[0])
        x_size = int(img_shape[1] * rand_scale[1]) 
        scale_factor = np.eye(4)
        scale_factor[0, 0] *= rand_scale[1]
        scale_factor[1, 1] *= rand_scale[0]
        results['img'] = [mmcv.imresize(img, (x_size, y_size), return_scale=False) for img in results['img']]
        lidar2img = [scale_factor @ l2i for l2i in results['lidar2img']]
        results['lidar2img'] = lidar2img
        results['img_shape'] = [img.shape for img in results['img']]
        return results

    def __repr__(self):
        repr_str = self.__class__.__name__
        repr_str += f'(size={self.scales}, '
        return repr_str


# @PIPELINES.register_module()
# class PhotoMetricDistortionMultiViewImage:
#     """Apply photometric distortion to image sequentially, every transformation
#     is applied with a probability of 0.5. The position of random contrast is in
#     second or second to last.
#     1. random brightness
#     2. random contrast (mode 0)
#     3. convert color from BGR to HSV
#     4. random saturation
#     5. random hue
#     6. convert color from HSV to BGR
#     7. random contrast (mode 1)
#     8. randomly swap channels
#     Args:
#         brightness_delta (int): delta of brightness.
#         contrast_range (tuple): range of contrast.
#         saturation_range (tuple): range of saturation.
#         hue_delta (int): delta of hue.
#     """

#     def __init__(self,
#                  brightness_delta=32,
#                  contrast_range=(0.5, 1.5),
#                  saturation_range=(0.5, 1.5),
#                  hue_delta=18):
#         self.brightness_delta = brightness_delta
#         self.contrast_lower, self.contrast_upper = contrast_range
#         self.saturation_lower, self.saturation_upper = saturation_range
#         self.hue_delta = hue_delta

#     def __call__(self, results):
#         """Call function to perform photometric distortion on images.
#         Args:
#             results (dict): Result dict from loading pipeline.
#         Returns:
#             dict: Result dict with images distorted.
#         """
#         imgs = results['img']
#         new_imgs = []
#         for img in imgs:
#             assert img.dtype == np.float32, \
#                 'PhotoMetricDistortion needs the input image of dtype np.float32,'\
#                 ' please set "to_float32=True" in "LoadImageFromFile" pipeline'
#             # random brightness
#             if random.randint(2):
#                 delta = random.uniform(-self.brightness_delta,
#                                     self.brightness_delta)
#                 img += delta

#             # mode == 0 --> do random contrast first
#             # mode == 1 --> do random contrast last
#             mode = random.randint(2)
#             if mode == 1:
#                 if random.randint(2):
#                     alpha = random.uniform(self.contrast_lower,
#                                         self.contrast_upper)
#                     img *= alpha

#             # convert color from BGR to HSV
#             img = mmcv.bgr2hsv(img)

#             # random saturation
#             if random.randint(2):
#                 img[..., 1] *= random.uniform(self.saturation_lower,
#                                             self.saturation_upper)

#             # random hue
#             if random.randint(2):
#                 img[..., 0] += random.uniform(-self.hue_delta, self.hue_delta)
#                 img[..., 0][img[..., 0] > 360] -= 360
#                 img[..., 0][img[..., 0] < 0] += 360

#             # convert color from HSV to BGR
#             img = mmcv.hsv2bgr(img)

#             # random contrast
#             if mode == 0:
#                 if random.randint(2):
#                     alpha = random.uniform(self.contrast_lower,
#                                         self.contrast_upper)
#                     img *= alpha

#             # randomly swap channels
#             if random.randint(2):
#                 img = img[..., random.permutation(3)]
#             new_imgs.append(img)
#         results['img'] = new_imgs
#         return results

#     def __repr__(self):
#         repr_str = self.__class__.__name__
#         repr_str += f'(\nbrightness_delta={self.brightness_delta},\n'
#         repr_str += 'contrast_range='
#         repr_str += f'{(self.contrast_lower, self.contrast_upper)},\n'
#         repr_str += 'saturation_range='
#         repr_str += f'{(self.saturation_lower, self.saturation_upper)},\n'
#         repr_str += f'hue_delta={self.hue_delta})'
#         return repr_str
    
@PIPELINES.register_module()
class LoadMultiViewImageFromFilesWaymo(object):
    """Load multi channel images from a list of separate channel files.
    Expects results['img_filename'] to be a list of filenames.
    Args:
        to_float32 (bool): Whether to convert the img to float32.
            Defaults to False.
        color_type (str): Color type of the file. Defaults to 'unchanged'.
    """

    def __init__(self, to_float32=False, img_scale=None, color_type='unchanged'):
        self.to_float32 = to_float32
        self.img_scale = img_scale
        self.color_type = color_type

    def pad(self, img):
        # to pad the 5 input images into a same size (for Waymo)
        if img.shape[0] != self.img_scale[0]:
            img = np.concatenate([img, np.zeros_like(img[0:1280-886,:])], axis=0)
        return img

    def __call__(self, results):
        """Call function to load multi-view image from files.
        Args:
            results (dict): Result dict containing multi-view image filenames.
        Returns:
            dict: The result dict containing the multi-view image data. \
                Added keys and values are described below.
                - filename (str): Multi-view image filenames.
                - img (np.ndarray): Multi-view image arrays.
                - img_shape (tuple[int]): Shape of multi-view image arrays.
                - ori_shape (tuple[int]): Shape of original image arrays.
                - pad_shape (tuple[int]): Shape of padded image arrays.
                - scale_factor (float): Scale factor.
                - img_norm_cfg (dict): Normalization configuration of images.
        """
        filename = results['img_filename']
        if self.img_scale is None:
            img = np.stack(
                [mmcv.imread(name, self.color_type) for name in filename], axis=-1)
        else:
            img = np.stack(
                [self.pad(mmcv.imread(name, self.color_type)) for name in filename], axis=-1)
        if self.to_float32:
            img = img.astype(np.float32)
        results['filename'] = filename
        # unravel to list, see `DefaultFormatBundle` in formating.py
        # which will transpose each image separately and then stack into array
        results['img'] = [img[..., i] for i in range(img.shape[-1])]
        results['img_shape'] = img.shape
        results['ori_shape'] = img.shape
        # Set initial values for default meta_keys
        results['pad_shape'] = img.shape
        # results['scale_factor'] = [1.0, 1.0]
        num_channels = 1 if len(img.shape) < 3 else img.shape[2]
        results['img_norm_cfg'] = dict(
            mean=np.zeros(num_channels, dtype=np.float32),
            std=np.ones(num_channels, dtype=np.float32),
            to_rgb=False)
        results['img_fields'] = ['img']
        return results

    def __repr__(self):
        """str: Return a string that describes the module."""
        return "{} (to_float32={}, color_type='{}')".format(
            self.__class__.__name__, self.to_float32, self.color_type)



# @PIPELINES.register_module()
# class MyResize(object):
#     """Resize images & bbox & mask.
#     This transform resizes the input image to some scale. Bboxes and masks are
#     then resized with the same scale factor. If the input dict contains the key
#     "scale", then the scale in the input dict is used, otherwise the specified
#     scale in the init method is used. If the input dict contains the key
#     "scale_factor" (if MultiScaleFlipAug does not give img_scale but
#     scale_factor), the actual scale will be computed by image shape and
#     scale_factor.
#     `img_scale` can either be a tuple (single-scale) or a list of tuple
#     (multi-scale). There are 3 multiscale modes:
#     - ``ratio_range is not None``: randomly sample a ratio from the ratio \
#       range and multiply it with the image scale.
#     - ``ratio_range is None`` and ``multiscale_mode == "range"``: randomly \
#       sample a scale from the multiscale range.
#     - ``ratio_range is None`` and ``multiscale_mode == "value"``: randomly \
#       sample a scale from multiple scales.
#     Args:
#         img_scale (tuple or list[tuple]): Images scales for resizing.
#         multiscale_mode (str): Either "range" or "value".
#         ratio_range (tuple[float]): (min_ratio, max_ratio)
#         keep_ratio (bool): Whether to keep the aspect ratio when resizing the
#             image.
#         bbox_clip_border (bool, optional): Whether clip the objects outside
#             the border of the image. Defaults to True.
#         backend (str): Image resize backend, choices are 'cv2' and 'pillow'.
#             These two backends generates slightly different results. Defaults
#             to 'cv2'.
#         override (bool, optional): Whether to override `scale` and
#             `scale_factor` so as to call resize twice. Default False. If True,
#             after the first resizing, the existed `scale` and `scale_factor`
#             will be ignored so the second resizing can be allowed.
#             This option is a work-around for multiple times of resize in DETR.
#             Defaults to False.
#     """

#     def __init__(self,
#                  img_scale=None,
#                  multiscale_mode='range',
#                  ratio_range=None,
#                  keep_ratio=True,
#                  bbox_clip_border=True,
#                  backend='cv2',
#                  override=False):
#         if img_scale is None:
#             self.img_scale = None
#         else:
#             if isinstance(img_scale, list):
#                 self.img_scale = img_scale
#             else:
#                 self.img_scale = [img_scale]
#             assert mmcv.is_list_of(self.img_scale, tuple)

#         if ratio_range is not None:
#             # mode 1: given a scale and a range of image ratio
#             assert len(self.img_scale) == 1
#         else:
#             # mode 2: given multiple scales or a range of scales
#             assert multiscale_mode in ['value', 'range']

#         self.backend = backend
#         self.multiscale_mode = multiscale_mode
#         self.ratio_range = ratio_range
#         self.keep_ratio = keep_ratio
#         # TODO: refactor the override option in Resize
#         self.override = override
#         self.bbox_clip_border = bbox_clip_border

#     @staticmethod
#     def random_select(img_scales):
#         """Randomly select an img_scale from given candidates.
#         Args:
#             img_scales (list[tuple]): Images scales for selection.
#         Returns:
#             (tuple, int): Returns a tuple ``(img_scale, scale_dix)``, \
#                 where ``img_scale`` is the selected image scale and \
#                 ``scale_idx`` is the selected index in the given candidates.
#         """

#         assert mmcv.is_list_of(img_scales, tuple)
#         scale_idx = np.random.randint(len(img_scales))
#         img_scale = img_scales[scale_idx]
#         return img_scale, scale_idx

#     @staticmethod
#     def random_sample(img_scales):
#         """Randomly sample an img_scale when ``multiscale_mode=='range'``.
#         Args:
#             img_scales (list[tuple]): Images scale range for sampling.
#                 There must be two tuples in img_scales, which specify the lower
#                 and uper bound of image scales.
#         Returns:
#             (tuple, None): Returns a tuple ``(img_scale, None)``, where \
#                 ``img_scale`` is sampled scale and None is just a placeholder \
#                 to be consistent with :func:`random_select`.
#         """

#         assert mmcv.is_list_of(img_scales, tuple) and len(img_scales) == 2
#         img_scale_long = [max(s) for s in img_scales]
#         img_scale_short = [min(s) for s in img_scales]
#         long_edge = np.random.randint(
#             min(img_scale_long),
#             max(img_scale_long) + 1)
#         short_edge = np.random.randint(
#             min(img_scale_short),
#             max(img_scale_short) + 1)
#         img_scale = (long_edge, short_edge)
#         return img_scale, None

#     @staticmethod
#     def random_sample_ratio(img_scale, ratio_range):
#         """Randomly sample an img_scale when ``ratio_range`` is specified.
#         A ratio will be randomly sampled from the range specified by
#         ``ratio_range``. Then it would be multiplied with ``img_scale`` to
#         generate sampled scale.
#         Args:
#             img_scale (tuple): Images scale base to multiply with ratio.
#             ratio_range (tuple[float]): The minimum and maximum ratio to scale
#                 the ``img_scale``.
#         Returns:
#             (tuple, None): Returns a tuple ``(scale, None)``, where \
#                 ``scale`` is sampled ratio multiplied with ``img_scale`` and \
#                 None is just a placeholder to be consistent with \
#                 :func:`random_select`.
#         """

#         assert isinstance(img_scale, tuple) and len(img_scale) == 2
#         min_ratio, max_ratio = ratio_range
#         assert min_ratio <= max_ratio
#         ratio = np.random.random_sample() * (max_ratio - min_ratio) + min_ratio
#         scale = int(img_scale[0] * ratio), int(img_scale[1] * ratio)
#         return scale, None

#     def _random_scale(self, results):
#         """Randomly sample an img_scale according to ``ratio_range`` and
#         ``multiscale_mode``.
#         If ``ratio_range`` is specified, a ratio will be sampled and be
#         multiplied with ``img_scale``.
#         If multiple scales are specified by ``img_scale``, a scale will be
#         sampled according to ``multiscale_mode``.
#         Otherwise, single scale will be used.
#         Args:
#             results (dict): Result dict from :obj:`dataset`.
#         Returns:
#             dict: Two new keys 'scale` and 'scale_idx` are added into \
#                 ``results``, which would be used by subsequent pipelines.
#         """

#         if self.ratio_range is not None:
#             scale, scale_idx = self.random_sample_ratio(
#                 self.img_scale[0], self.ratio_range)
#         elif len(self.img_scale) == 1:
#             scale, scale_idx = self.img_scale[0], 0
#         elif self.multiscale_mode == 'range':
#             scale, scale_idx = self.random_sample(self.img_scale)
#         elif self.multiscale_mode == 'value':
#             scale, scale_idx = self.random_select(self.img_scale)
#         else:
#             raise NotImplementedError

#         results['scale'] = scale
#         results['scale_idx'] = scale_idx

#     def _resize_img(self, results):
#         """Resize images with ``results['scale']``."""
#         imgs = results['img']
#         results['img'] = [imgs[i] for i in range(len(imgs))]
#         for key in results.get('img_fields', ['img']):
#             for idx in range(len(results['img'])):
#                 if self.keep_ratio:
#                     img, scale_factor = mmcv.imrescale(
#                         results[key][idx],
#                         results['scale'],
#                         return_scale=True,
#                         backend=self.backend)
#                     # the w_scale and h_scale has minor difference
#                     # a real fix should be done in the mmcv.imrescale in the future
#                     new_h, new_w = img.shape[:2]
#                     h, w = results[key][idx].shape[:2]
#                     w_scale = new_w / w
#                     h_scale = new_h / h
#                 else:
#                     img, w_scale, h_scale = mmcv.imresize(
#                         results[key][idx],
#                         results['scale'],
#                         return_scale=True,
#                         backend=self.backend)
#                 results[key][idx] = img

#             scale_factor = np.array([w_scale, h_scale, w_scale, h_scale],
#                                     dtype=np.float32)
#             results['img_shape'] = img.shape
#             # in case that there is no padding
#             results['pad_shape'] = img.shape
#             results['scale_factor'] = scale_factor
#             results['keep_ratio'] = self.keep_ratio

#     def _resize_bboxes(self, results):
#         """Resize bounding boxes with ``results['scale_factor']``."""
#         for key in results.get('bbox_fields', []):
#             bboxes = results[key] * results['scale_factor']
#             if self.bbox_clip_border:
#                 img_shape = results['img_shape']
#                 bboxes[:, 0::2] = np.clip(bboxes[:, 0::2], 0, img_shape[1])
#                 bboxes[:, 1::2] = np.clip(bboxes[:, 1::2], 0, img_shape[0])
#             results[key] = bboxes

#     def _resize_masks(self, results):
#         """Resize masks with ``results['scale']``"""
#         for key in results.get('mask_fields', []):
#             if results[key] is None:
#                 continue
#             if self.keep_ratio:
#                 results[key] = results[key].rescale(results['scale'])
#             else:
#                 results[key] = results[key].resize(results['img_shape'][:2])

#     def _resize_seg(self, results):
#         """Resize semantic segmentation map with ``results['scale']``."""
#         for key in results.get('seg_fields', []):
#             if self.keep_ratio:
#                 gt_seg = mmcv.imrescale(
#                     results[key],
#                     results['scale'],
#                     interpolation='nearest',
#                     backend=self.backend)
#             else:
#                 gt_seg = mmcv.imresize(
#                     results[key],
#                     results['scale'],
#                     interpolation='nearest',
#                     backend=self.backend)
#             results['gt_semantic_seg'] = gt_seg

#     def __call__(self, results):
#         """Call function to resize images, bounding boxes, masks, semantic
#         segmentation map.
#         Args:
#             results (dict): Result dict from loading pipeline.
#         Returns:
#             dict: Resized results, 'img_shape', 'pad_shape', 'scale_factor', \
#                 'keep_ratio' keys are added into result dict.
#         """

#         if 'scale' not in results:
#             if 'scale_factor' in results:
#                 img_shape = results['img'][0].shape[:2]
#                 scale_factor = results['scale_factor']
#                 assert isinstance(scale_factor, float)
#                 results['scale'] = tuple(
#                     [int(x * scale_factor) for x in img_shape][::-1])
#             else:
#                 self._random_scale(results)
#         else:
#             if not self.override:
#                 assert 'scale_factor' not in results, (
#                     'scale and scale_factor cannot be both set.')
#             else:
#                 results.pop('scale')
#                 if 'scale_factor' in results:
#                     results.pop('scale_factor')
#                 self._random_scale(results)

#         self._resize_img(results)
#         self._resize_bboxes(results)
#         self._resize_masks(results)
#         self._resize_seg(results)
#         return results

#     def __repr__(self):
#         repr_str = self.__class__.__name__
#         repr_str += f'(img_scale={self.img_scale}, '
#         repr_str += f'multiscale_mode={self.multiscale_mode}, '
#         repr_str += f'ratio_range={self.ratio_range}, '
#         repr_str += f'keep_ratio={self.keep_ratio}, '
#         repr_str += f'bbox_clip_border={self.bbox_clip_border})'
#         return repr_str


# @PIPELINES.register_module()
# class MyNormalize(object):
#     """Normalize the image.
#     Added key is "img_norm_cfg".
#     Args:
#         mean (sequence): Mean values of 3 channels.
#         std (sequence): Std values of 3 channels.
#         to_rgb (bool): Whether to convert the image from BGR to RGB,
#             default is true.
#     """

#     def __init__(self, mean, std, to_rgb=True):
#         self.mean = np.array(mean, dtype=np.float32)
#         self.std = np.array(std, dtype=np.float32)
#         self.to_rgb = to_rgb

#     def __call__(self, results):
#         """Call function to normalize images.
#         Args:
#             results (dict): Result dict from loading pipeline.
#         Returns:
#             dict: Normalized results, 'img_norm_cfg' key is added into
#                 result dict.
#         """
#         for key in results.get('img_fields', ['img']):
#             for idx in range(len(results['img'])):
#                 results[key][idx] = mmcv.imnormalize(results[key][idx], self.mean, self.std,
#                                                      self.to_rgb)
#         results['img_norm_cfg'] = dict(
#             mean=self.mean, std=self.std, to_rgb=self.to_rgb)
#         return results

#     def __repr__(self):
#         repr_str = self.__class__.__name__
#         repr_str += f'(mean={self.mean}, std={self.std}, to_rgb={self.to_rgb})'
#         return repr_str


# @PIPELINES.register_module()
# class MyPad(object):
#     """Pad the image & mask.
#     There are two padding modes: (1) pad to a fixed size and (2) pad to the
#     minimum size that is divisible by some number.
#     Added keys are "pad_shape", "pad_fixed_size", "pad_size_divisor",
#     Args:
#         size (tuple, optional): Fixed padding size.
#         size_divisor (int, optional): The divisor of padded size.
#         pad_val (float, optional): Padding value, 0 by default.
#     """

#     def __init__(self, size=None, size_divisor=None, pad_val=0):
#         self.size = size
#         self.size_divisor = size_divisor
#         self.pad_val = pad_val
#         # only one of size and size_divisor should be valid
#         assert size is not None or size_divisor is not None
#         assert size is None or size_divisor is None

#     def _pad_img(self, results):
#         """Pad images according to ``self.size``."""
#         for key in results.get('img_fields', ['img']):
#             if self.size is not None:
#                 padded_img = mmcv.impad(
#                     results[key], shape=self.size, pad_val=self.pad_val)
#             elif self.size_divisor is not None:
#                 for idx in range(len(results[key])):
#                     padded_img = mmcv.impad_to_multiple(
#                         results[key][idx], self.size_divisor, pad_val=self.pad_val)
#                     results[key][idx] = padded_img
#         results['pad_shape'] = padded_img.shape
#         results['pad_fixed_size'] = self.size
#         results['pad_size_divisor'] = self.size_divisor

#     def _pad_masks(self, results):
#         """Pad masks according to ``results['pad_shape']``."""
#         pad_shape = results['pad_shape'][:2]
#         for key in results.get('mask_fields', []):
#             results[key] = results[key].pad(pad_shape, pad_val=self.pad_val)

#     def _pad_seg(self, results):
#         """Pad semantic segmentation map according to
#         ``results['pad_shape']``."""
#         for key in results.get('seg_fields', []):
#             results[key] = mmcv.impad(
#                 results[key], shape=results['pad_shape'][:2])

#     def __call__(self, results):
#         """Call function to pad images, masks, semantic segmentation maps.
#         Args:
#             results (dict): Result dict from loading pipeline.
#         Returns:
#             dict: Updated result dict.
#         """
#         self._pad_img(results)
#         self._pad_masks(results)
#         self._pad_seg(results)
#         return results

#     def __repr__(self):
#         repr_str = self.__class__.__name__
#         repr_str += f'(size={self.size}, '
#         repr_str += f'size_divisor={self.size_divisor}, '
#         repr_str += f'pad_val={self.pad_val})'
#         return repr_str
    
# @PIPELINES.register_module()
# class MyFlip3D(RandomFlip):
#     """Flip the points & bbox.

#     If the input dict contains the key "flip", then the flag will be used,
#     otherwise it will be randomly decided by a ratio specified in the init
#     method.

#     Args:
#         sync_2d (bool, optional): Whether to apply flip according to the 2D
#             images. If True, it will apply the same flip as that to 2D images.
#             If False, it will decide whether to flip randomly and independently
#             to that of 2D images. Defaults to True.
#         flip_ratio_bev_horizontal (float, optional): The flipping probability
#             in horizontal direction. Defaults to 0.0.
#         flip_ratio_bev_vertical (float, optional): The flipping probability
#             in vertical direction. Defaults to 0.0.
#     """

#     def __init__(self,
#                  sync_2d=True,
#                  flip_ratio_bev_horizontal=0.0,
#                  flip_ratio_bev_vertical=0.0,
#                  **kwargs):
#         super(MyFlip3D, self).__init__(
#             flip_ratio=flip_ratio_bev_horizontal, **kwargs)
#         self.sync_2d = sync_2d
#         self.flip_ratio_bev_vertical = flip_ratio_bev_vertical
#         if flip_ratio_bev_horizontal is not None:
#             assert isinstance(
#                 flip_ratio_bev_horizontal,
#                 (int, float)) and 0 <= flip_ratio_bev_horizontal <= 1
#         if flip_ratio_bev_vertical is not None:
#             assert isinstance(
#                 flip_ratio_bev_vertical,
#                 (int, float)) and 0 <= flip_ratio_bev_vertical <= 1

#     def random_flip_data_3d(self, input_dict, direction='horizontal'):
#         """Flip 3D data randomly.

#         Args:
#             input_dict (dict): Result dict from loading pipeline.
#             direction (str): Flip direction. Default: horizontal.

#         Returns:
#             dict: Flipped results, 'points', 'bbox3d_fields' keys are \
#                 updated in the result dict.
#         """
#         assert direction in ['horizontal', 'vertical']
#         if len(input_dict['bbox3d_fields']) == 0:  # test mode
#             input_dict['bbox3d_fields'].append('empty_box3d')
#             input_dict['empty_box3d'] = input_dict['box_type_3d'](
#                 np.array([], dtype=np.float32))
#         assert len(input_dict['bbox3d_fields']) == 1
#         for key in input_dict['bbox3d_fields']:
#             if 'points' in input_dict:
#                 input_dict['points'] = input_dict[key].flip(
#                     direction, points=input_dict['points'])
#             else:
#                 input_dict[key].flip(direction)
#         if 'centers2d' in input_dict:
#             assert self.sync_2d is True and direction == 'horizontal', \
#                 'Only support sync_2d=True and horizontal flip with images'
#             w = input_dict['ori_shape'][1]
#             input_dict['centers2d'][..., 0] = \
#                 w - input_dict['centers2d'][..., 0]
#             # need to modify the horizontal position of camera center
#             # along u-axis in the image (flip like centers2d)
#             # ['cam2img'][0][2] = c_u
#             # see more details and examples at
#             # https://github.com/open-mmlab/mmdetection3d/pull/744
#             input_dict['cam2img'][0][2] = w - input_dict['cam2img'][0][2]

#     def __call__(self, input_dict):
#         """Call function to flip points, values in the ``bbox3d_fields`` and \
#         also flip 2D image and its annotations.

#         Args:
#             input_dict (dict): Result dict from loading pipeline.

#         Returns:
#             dict: Flipped results, 'flip', 'flip_direction', \
#                 'pcd_horizontal_flip' and 'pcd_vertical_flip' keys are added \
#                 into result dict.
#         """
#         # filp 2D image and its annotations
#         super(MyFlip3D, self).__call__(input_dict)
#         input_dict['img'] = [img for img in input_dict['img']]
#         if self.sync_2d:
#             input_dict['pcd_horizontal_flip'] = input_dict['flip']
#             input_dict['pcd_vertical_flip'] = False
#         else:
#             if 'pcd_horizontal_flip' not in input_dict:
#                 flip_horizontal = True if np.random.rand(
#                 ) < self.flip_ratio else False
#                 input_dict['pcd_horizontal_flip'] = flip_horizontal
#             if 'pcd_vertical_flip' not in input_dict:
#                 flip_vertical = True if np.random.rand(
#                 ) < self.flip_ratio_bev_vertical else False
#                 input_dict['pcd_vertical_flip'] = flip_vertical

#         if 'transformation_3d_flow' not in input_dict:
#             input_dict['transformation_3d_flow'] = []

#         if input_dict['pcd_horizontal_flip']:
#             self.random_flip_data_3d(input_dict, 'horizontal')
#             input_dict['transformation_3d_flow'].extend(['HF'])
#         if input_dict['pcd_vertical_flip']:
#             self.random_flip_data_3d(input_dict, 'vertical')
#             input_dict['transformation_3d_flow'].extend(['VF'])
#         return input_dict

#     def __repr__(self):
#         """str: Return a string that describes the module."""
#         repr_str = self.__class__.__name__
#         repr_str += f'(sync_2d={self.sync_2d},'
#         repr_str += f' flip_ratio_bev_vertical={self.flip_ratio_bev_vertical})'
#         return repr_str
