"""
A: CONV BN ACT structure
B: BN ACT CONV structure
C: ACT BN CONV structure
D: ACT CONV BN structure
"""

import torch
import torch.nn as nn


class BasicResidualV1A(nn.Module):
    def __init__(self, in_planes, planes, stride=1):
        super(BasicResidualV1A, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.act1 = nn.ReLU(inplace=True)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.act2 = nn.ReLU(inplace=True)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(planes)
            )

    def forward(self, x):
        out = self.conv1(x)
        out = self.bn1(out)
        out = self.act1(out)
        out = self.conv2(out)
        out = self.bn2(out)
        out += self.shortcut(x)
        out = self.act2(out)
        return out


class BasicResidualV1B(nn.Module):

    def __init__(self, in_planes, planes, stride=1):
        super(BasicResidualV1B, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.act1 = nn.ReLU(inplace=True)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.act2 = nn.ReLU(inplace=True)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(planes)
            )

    def forward(self, x):
        out = self.bn1(x)
        out = self.act1(out)
        out = self.conv1(out)
        out = self.bn2(out)
        out = self.act2(out)
        out = self.conv2(out)
        out += self.shortcut(x)
        return out


class BasicResidualV1C(nn.Module):

    def __init__(self, in_planes, planes, stride=1):
        super(BasicResidualV1C, self).__init__()
        self.relu1 = nn.ReLU(inplace=True)
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.relu2 = nn.ReLU(inplace=True)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(planes)
            )

    def forward(self, x):
        out = self.relu1(x)
        out = self.bn1(out)
        out = self.conv1(out)
        out = self.relu2(out)
        out = self.bn2(out)
        out = self.conv2(out)
        out += self.shortcut(x)
        return out


class BasicResidualV1D(nn.Module):

    def __init__(self, in_planes, planes, stride=1):
        super(BasicResidualV1D, self).__init__()
        self.relu1 = nn.ReLU(inplace=True)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.relu2 = nn.ReLU(inplace=True)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(planes)
            )

    def forward(self, x):
        out = self.relu1(x)
        out = self.conv1(out)
        out = self.bn1(out)
        out = self.relu2(out)
        out = self.conv2(out)
        out = self.bn2(out)
        out += self.shortcut(x)
        return out


class BasicResidualV2A(nn.Module):
    def __init__(self, in_channel, out_channel, stride=1):
        super().__init__()
        self.hidden_channel = out_channel // 4
        self.conv1 = nn.Conv2d(in_channel, self.hidden_channel, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(self.hidden_channel)
        self.relu1 = nn.ReLU()
        self.conv2 = nn.Conv2d(self.hidden_channel, self.hidden_channel, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(self.hidden_channel)
        self.relu2 = nn.ReLU(inplace=True)
        self.conv3 = nn.Conv2d(self.hidden_channel, out_channel, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(out_channel)
        self.relu3 = nn.ReLU(inplace=True)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_channel != out_channel:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_channel, out_channel, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(out_channel),
            )

    def forward(self, x):
        shortcut = self.shortcut(x)
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu1(x)
        x = self.conv2(x)
        x = self.bn2(x)
        x = self.relu2(x)
        x = self.conv3(x)
        x = self.bn3(x)
        x = x + shortcut
        x = self.relu3(x)
        return x


class BasicResidualV2B(nn.Module):
    def __init__(self, in_channel, out_channel, stride=1):
        super().__init__()
        self.hidden_channel = out_channel // 4
        self.bn1 = nn.BatchNorm2d(in_channel)
        self.relu1 = nn.ReLU()
        self.conv1 = nn.Conv2d(in_channel, self.hidden_channel, kernel_size=1, bias=False)
        self.bn2 = nn.BatchNorm2d(self.hidden_channel)
        self.relu2 = nn.ReLU(inplace=True)
        self.conv2 = nn.Conv2d(self.hidden_channel, self.hidden_channel, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.hidden_channel)
        self.relu3 = nn.ReLU(inplace=True)
        self.conv3 = nn.Conv2d(self.hidden_channel, out_channel, kernel_size=1, bias=False)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_channel != out_channel:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_channel, out_channel, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(out_channel),
            )

    def forward(self, x):
        shortcut = self.shortcut(x)
        x = self.bn1(x)
        x = self.relu1(x)
        x = self.conv1(x)
        x = self.bn2(x)
        x = self.relu2(x)
        x = self.conv2(x)
        x = self.bn3(x)
        x = self.relu3(x)
        x = self.conv3(x)
        x += shortcut
        return x


class BasicResidualV2C(nn.Module):
    def __init__(self, in_channel, out_channel, stride=1):
        super().__init__()
        self.hidden_channel = out_channel // 4
        self.relu1 = nn.ReLU()
        self.bn1 = nn.BatchNorm2d(in_channel)
        self.conv1 = nn.Conv2d(in_channel, self.hidden_channel, kernel_size=1, bias=False)
        self.relu2 = nn.ReLU(inplace=True)
        self.bn2 = nn.BatchNorm2d(self.hidden_channel)
        self.conv2 = nn.Conv2d(self.hidden_channel, self.hidden_channel, kernel_size=3, stride=stride, padding=1, bias=False)
        self.relu3 = nn.ReLU(inplace=True)
        self.bn3 = nn.BatchNorm2d(self.hidden_channel)
        self.conv3 = nn.Conv2d(self.hidden_channel, out_channel, kernel_size=1, bias=False)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_channel != out_channel:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_channel, out_channel, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(out_channel),
            )

    def forward(self, x):
        shortcut = self.shortcut(x)
        x = self.relu1(x)
        x = self.bn1(x)
        x = self.conv1(x)
        x = self.relu2(x)
        x = self.bn2(x)
        x = self.conv2(x)
        x = self.relu3(x)
        x = self.bn3(x)
        x = self.conv3(x)
        x += shortcut
        return x


class BasicResidualV2D(nn.Module):
    def __init__(self, in_channel, out_channel, stride=1):
        super().__init__()
        self.hidden_channel = out_channel // 4
        self.relu1 = nn.ReLU()
        self.conv1 = nn.Conv2d(in_channel, self.hidden_channel, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(self.hidden_channel)
        self.relu2 = nn.ReLU(inplace=True)
        self.conv2 = nn.Conv2d(self.hidden_channel, self.hidden_channel, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(self.hidden_channel)
        self.relu3 = nn.ReLU(inplace=True)
        self.conv3 = nn.Conv2d(self.hidden_channel, out_channel, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(out_channel)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_channel != out_channel:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_channel, out_channel, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(out_channel),
            )

    def forward(self, x):
        shortcut = self.shortcut(x)
        x = self.relu1(x)
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu2(x)
        x = self.conv2(x)
        x = self.bn2(x)
        x = self.relu3(x)
        x = self.conv3(x)
        x = self.bn3(x)
        x += shortcut
        return x


class BasicBlock(nn.Module):
    def __init__(self, module, in_channel, out_channel, stride, n_stack):
        super(BasicBlock, self).__init__()
        self.in_channel = in_channel
        self.out_channel = out_channel
        self.stride = stride
        self.module = module
        self.n_stack = n_stack

        self.layers = nn.ModuleList()
        for i in range(n_stack):
            self.layers.append(module(in_channel, out_channel, stride))
            in_channel = out_channel
            stride = 1
        self.layers = nn.Sequential(*self.layers)

    def forward(self, x):
        return self.layers(x)


class ResNetV1(nn.Module):
    def __init__(self, module, model_shape, num_classes):
        super(ResNetV1, self).__init__()
        self.model_shape = model_shape
        self.num_classes = num_classes
        self.in_planes = 64
        self.conv1 = nn.Conv2d(3, 64, kernel_size=7, stride=2, padding=3, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = BasicBlock(module, 64, 64, 1, model_shape[0])
        self.layer2 = BasicBlock(module, 64, 128, 2, model_shape[1])
        self.layer3 = BasicBlock(module, 128, 256, 2, model_shape[2])
        self.layer4 = BasicBlock(module, 256, 512, 2, model_shape[3])
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(512, num_classes)

    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)
        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)
        x = self.avgpool(x)
        x = x.view(x.size(0), -1)
        x = self.fc(x)
        return x


class ResNetV2(nn.Module):
    def __init__(self, module, model_shape, num_classes):
        super(ResNetV2, self).__init__()
        self.model_shape = model_shape
        self.num_classes = num_classes
        self.in_planes = 64
        self.conv1 = nn.Conv2d(3, 64, kernel_size=7, stride=2, padding=3, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.relu = nn.ReLU(inplace=True)
        self.layer1 = BasicBlock(module, 64, 256, 1, model_shape[0])
        self.layer2 = BasicBlock(module, 256, 512, 2, model_shape[1])
        self.layer3 = BasicBlock(module, 512, 1024, 2, model_shape[2])
        self.layer4 = BasicBlock(module, 1024, 2048, 2, model_shape[3])
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(2048, num_classes)

    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)
        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)
        x = self.avgpool(x)
        x = x.view(x.size(0), -1)
        x = self.fc(x)
        return x
