"""
模拟退火算法

优化目标loss，初始值x0，温度T，温度衰减系数T_alpha，最低温度T_min，随机因子beta。

在当前参数上进行扰动，计算新的loss值。
两种情况：
如果新的loss更小，那么接收当前参数（此时降温）；
如果新的loss更大，那么以一定的概率(exp(-d/T))接收当前参数（为了能够跳出局部最优）。

"""

import numpy as np


class SimulatedAnnealing:
    def __init__(self, min_f: callable,
                 x0: np.array,
                 x_low_bounds: np.array,
                 x_up_bounds: np.array,
                 noise_ratio=0.2,
                 T=1e5, T_min=1e-3, T_alpha=0.99,
                 max_trial=1000,
                 ):
        assert x0.ndim == x_low_bounds.ndim == x_up_bounds.ndim == 1
        assert x0.size == x_low_bounds.size == x_up_bounds.size
        assert T > T_min, T_alpha < 1

        self._min_f = min_f
        self._x0 = x0
        self._x_low_bounds = x_low_bounds
        self._x_up_bounds = x_up_bounds
        self._noise_ratio = noise_ratio
        self._T = T
        self._T_min = T_min
        self._T_alpha = T_alpha
        self._max_trial = max_trial

    def annealing(self):
        T = self._T

        old_x = self._x0
        old_val = self._min_f(self._x0)

        best_x = None
        best_val = float('inf')

        cnt = 0

        cnt2 = 0
        while T > self._T_min:
            cnt2 += 1
            if cnt2 % 1000 == 0:
                # print(cnt2, old_x, T)
                pass

            new_x = old_x + self._noise(old_x)

            new_val = self._min_f(new_x)

            if new_val < best_val:
                best_x, best_val = new_x, new_val

            if new_val < old_val:
                old_x = new_x
                old_val = new_val
                T = T * self._T_alpha
            elif np.random.rand() < np.exp(-(new_val - old_val) / T):
                old_x = new_x
                old_val = new_val
            else:
                cnt += 1
                # print("cnt:", cnt)

            if cnt > self._max_trial:
                break

        return best_x

    def _noise(self, x: np.array) -> np.array:
        """
        对参数进行各维独立的均匀随机扰动。
        扰动半径是预先定义的常数，
        radius = (u - l) / 5
        """
        radius = (self._x_up_bounds - self._x_low_bounds) * self._noise_ratio
        noise = 2 * (np.random.rand() - 0.5) * radius

        for i in range(x.size):
            if x[i] + noise[i] > self._x_up_bounds[i] or x[i] + noise[i] < self._x_low_bounds[i]:
                noise[i] = -noise[i]
        return noise


def f(x: np.array):
    return x[0] + 10 * np.sin(5 * x[0]) + 7 * np.cos(4 * x[0])


if __name__ == "__main__":
    sa = SimulatedAnnealing(f,
                            np.array([4.0]),
                            np.array([0.0]),
                            np.array([9.0]),
                            )
    x = sa.annealing()
    print("x:", x, "f:", f(x))
