from PIL import Image
import numpy as np

FRUIT = ['apple']
COOKABLE = ['lettuce', 'onion', 'potato', 'tomato']
SLICEABLE = ['apple', 'lettuce', 'onion', 'potato', 'tomato']
COOKER = ['pot', 'pan']
CUTTER = ['knife']
CLEANER = ['sink']
HEATER = ['stove']
SERVABLE = ['plate']
DIRTYABLE = ['pot', 'pan', 'plate']
AGENT = ['agent']
ALL_OBJS = SLICEABLE + COOKER + CUTTER + CLEANER + HEATER + SERVABLE + AGENT
OBJ_TO_IDX = {obj: i for i, obj in enumerate(ALL_OBJS)}

OPTIONS = ['pick', 'place', 'slice', 'clean', 'cook', 'goto', 'done']
EOS_TOKEN  = 'Done'
OBJ_DEFAULT_STATE = {
    'dirty': False,
    'sliced': False,
    'holding': False,
    'cooked': False,
    'on': None,
    'loc': None
}

VOCAB = ALL_OBJS + OPTIONS + ['the', 'go', 'to', 'up']
VOCAB_TO_IDX = {w: i for i, w in enumerate(VOCAB)}

def state_to_vec(state, simplified=False, target_obj=None, rooms=0):
    if simplified:
        is_sliced = state[target_obj]['sliced']
        is_pan = 'pan' in state
        assert (is_pan and (not 'pot' in state)) or ((not is_pan) and ('pot' in state))
        is_dirty = state[('pan' if is_pan else 'pot')]['dirty']
        vec = np.zeros(8, dtype=np.float32)
        vec[is_sliced * 4 + is_pan * 2 + is_dirty] = 1
        return vec
    else:
        vec = np.zeros(len(ALL_OBJS) * (3+rooms), dtype=np.float32)
        for obj, conds in state.items():
            idx = (3+rooms) * OBJ_TO_IDX[obj]
            vec[idx] = 1
            if conds['sliced']:
                vec[idx + 1] = 1
            if conds['dirty']:
                vec[idx + 2] = 1
            if rooms > 0:
                offset = 3 + (ord(conds['room']) - ord('A'))
                assert 2 < offset < 3 + rooms
                vec[idx + offset] = 1
        return vec

def op_cook(a, state):
    assert a in COOKABLE
    name = f'Cook the {a}.'
    if state[a]['cooked']:
        return []
    seq = op_slice(a, state)
    if 'pot' in state:
        seq.extend(op_clean('pot', state))
        seq.extend(op_place(a, 'pot', state))
        seq.extend(op_place('pot', 'stove', state))
        seq.append(f'Boil the {a}.')
    elif 'pan' in state:
        seq.extend(op_clean('pan', state))
        seq.extend(op_place(a, 'pan', state))
        seq.extend(op_place('pan', 'stove', state))
        seq.append(f'Fry the {a}.')
    else:
        raise ValueError('No pot or pan for cooking')
    state[a]['cooked'] = True

    seq.extend(op_done(state))
    return seq

def op_done(state):
    return [EOS_TOKEN]

def op_pick(a, state):
    if state[a]['holding']:
        return []
    seq = op_goto(state[a]['room'], state)
    seq.append(f'Pick up the {a}.')

    state[a]['on'] = None
    return seq

def op_place(a, b, state):
    if state[a]['on'] == b:
        return []
    seq = op_pick(a, state)
    seq.extend(op_goto(state[b]['room'], state))
    seq.append(f'Place the {a} on the {b}.')

    state[a]['on'] = b
    return seq

def op_slice(a, state):
    if state[a]['sliced']:
        return []
    seq = op_pick('knife', state)
    seq.extend(op_goto(state[a]['room'], state))
    seq.append(f'Slice the {a}')

    state[a]['sliced'] = True
    return seq

def op_clean(a, state):
    if not state[a]['dirty']:
        return []
    seq = op_place(a, 'sink', state)
    seq.append(f'Clean the {a}')

    state[a]['dirty'] = False
    return seq

def op_goto(room, state):
    if state['agent']['room'] == room:
        return []
    state['agent']['room'] = room
    return [f'Go to room {room}.']

OPTION_TO_OP = {
    'goto': op_goto,
    'pick': op_pick,
    'place': op_place,
    'slice': op_slice,
    'clean': op_clean,
    'cook': op_cook,
    'done': op_done
}

