import random
import time
import requests
import json
from openai import OpenAI
from google import genai
import anthropic
import httpx


class ChatAPI:
    def __init__(
        self,
        openai_api_key=None,
        deepseek_api_key=None,
        gemini_api_key=None,
        grok_api_key=None,
        llama_api_key=None,
        claude_api_key=None,
        qwen_api_key=None,
        max_tokens=8192
    ):
        # API keys
        self.api_keys = {
            "openai": openai_api_key,
            "deepseek": deepseek_api_key,
            "gemini": gemini_api_key,
            "grok": grok_api_key,
            "llama": llama_api_key,
            "claude": claude_api_key,
            "qwen": qwen_api_key,
        }
        self.endpoints = {
            "openai": "https://api.openai.com/v1/chat/completions",
            "deepseek": "https://api.deepseek.com/v1/chat/completions",
            "gemini": "https://api.gemini.com/v1/chat/completions",
            "grok": "https://api.x.ai/v1",
            "llama": "https://api.llama.com/v1/chat/completions",
            "claude": "https://api.anthropic.com/v1/messages",
        }
        self.max_tokens = max_tokens
        self.grok = OpenAI(
            api_key=grok_api_key,
            base_url="https://api.x.ai/v1",
            timeout=60
        )
        self.gemini = genai.Client(api_key=gemini_api_key)
        self.llama = OpenAI(
            api_key=llama_api_key,
            base_url="https://api.llmapi.com",
            timeout=60
        )
        self.qwen = OpenAI(
            api_key=qwen_api_key,
            base_url="https://dashscope.aliyuncs.com/compatible-mode/v1",
            timeout=60
        )

        self.claude = anthropic.Anthropic(
            api_key=claude_api_key,
        )
        

    def _post(self, service, payload):
        api_key = self.api_keys.get(service)
        if not api_key:
            raise ValueError(f"No API key provided for {service!r}")
        url = self.endpoints[service]
        headers = {
            "Content-Type": "application/json",
            "Authorization": f"Bearer {api_key}"
        }
        sess = requests.Session()
        sess.keep_alive = False
        resp = requests.post(url, headers=headers, json=payload, timeout=1200, proxies=proxies)
        resp.raise_for_status()
        data = resp.json()
         # OpenAI / Deepseek (choices/message)
        if "choices" in data:
            return data["choices"][0]["message"]["content"]

        if "content" in data:
            parts = data["content"]
            return "".join(chunk.get("text", "") for chunk in parts if chunk.get("type") == "text")

        if "completion" in data:
            return data["completion"]

    def call_model(self, messages, model, temperature=0.7, max_tokens=8192):
        """
        Generic single-entry point: detects service by model name.
        """
        max_tokens = max_tokens or self.max_tokens
        # choose service by model prefix
        lower = model.lower()
        if lower.startswith(("gpt", "gpt-4", "gpt-3")):
            service = "openai"
        elif lower.startswith("deepseek"):
            service = "deepseek"
        elif lower.startswith("gemini"):
            service = "gemini"
        elif lower.startswith("grok"):
            service = "grok"
        elif lower.startswith("llama"):
            service = "llama"
        elif lower.startswith("claude"):
            service = "claude"
        else:
            raise ValueError(f"Unknown model: {model}")

        payload = {
            "model": model,
            "messages": messages,
            "temperature": temperature,
            "max_tokens": max_tokens,
        }
        return self._post(service, payload)

    # wrappers
    def call_gpt(self, messages, model="gpt-3.5-turbo", temperature=0.7, max_tokens=8192):
        return self.call_model(messages, model, temperature, max_tokens)

    def call_deepseek(self, messages, model="deepseek-chat", temperature=0.7, max_tokens=8192):
        return self.call_model(messages, model, temperature, max_tokens)

    def call_gemini(self, messages, model="gemini-pro", temperature=0.7, max_tokens=8192):
        response = self.gemini.models.generate_content(
            model=model, 
            contents=messages[0]['content'],
            temperature=temperature,
            max_tokens=max_tokens
        )
        return response.text

    def call_grok(self, messages, model="grok-3", temperature=0.7, max_tokens=8192):
        completion = self.grok.chat.completions.create(
            model = model,
            messages=messages,
            max_tokens=max_tokens,
            temperature=temperature
        )
        return completion.choices[0].message.content

    def call_llama(self, messages, model="llama4-maverick", temperature=0.7, max_tokens=None):
        completion = self.llama.chat.completions.create(
            model=model,
            messages=messages,
            temperature=temperature,
            max_tokens=max_tokens
        )
        return completion.choices[0].message.content

    def call_claude(self, messages, model="claude-3", temperature=0.7, max_tokens=8192):
        res = ''
        with self.claude.messages.stream(
            max_tokens=max_tokens,
            messages=messages,
            model=model,
            temperature=temperature
        ) as stream:
            for text in stream.text_stream:
                res += text
        
        return res
    
    def call_qwen(self, messages, model="qwen-plus", temperature=0.7, max_tokens=8192):
        completion = self.qwen.chat.completions.create(
            model=model,
            messages=messages,
            max_tokens=max_tokens,
            temperature=temperature
        )
        return completion.choices[0].message.content
