import inspect
import time
from dataclasses import dataclass, field
from typing import Dict, List, Callable, Any
import uuid

@dataclass
class FunctionIndividual:
    func: Callable
    metadata: Dict[str, Any] = field(default_factory=dict)
    individual_id: str = field(default_factory=lambda: str(uuid.uuid4()))
    created_at: float = field(default_factory=time.time)
    
    @property
    def source_code(self) -> str:
        return inspect.getsource(self.func)
    
    @property
    def func_name(self) -> str:
        return self.func.__name__
    
    def __call__(self, *args, **kwds):
        return self.func(*args, **kwds)

class Population:
    def __init__(self):
        self._population: Dict[str, FunctionIndividual] = {}
        self._name_index: Dict[str, List[str]] = {}
    
    def add_function(self, 
                    func: Callable, 
                    metadata: Dict = None) -> str:
        individual = FunctionIndividual(
            func=func,
            metadata=metadata or {}
        )
        
        self._population[individual.individual_id] = individual
        self._name_index.setdefault(individual.func_name, []).append(individual.individual_id)
        
        return individual.individual_id

    def get_by_id(self, individual_id: str) -> FunctionIndividual:
        return self._population.get(individual_id)

    def search_by_name(self, name: str) -> List[FunctionIndividual]:
        return [self._population[iid] for iid in self._name_index.get(name, [])]

    def search_by_source_pattern(self, pattern: str) -> List[FunctionIndividual]:
        results = []
        for individual in self._population.values():
            if pattern in individual.source_code:
                results.append(individual)
        return results

    def filter_by_metadata(self, filters) -> List[FunctionIndividual]:
        results = []
        for individual in self._population.values():
            match = True
            for key, value in filters.items():
                if individual.metadata.get(key) != value:
                    match = False
                    break
            if match:
                results.append(individual)
        return results

    @property
    def size(self) -> int:
        return len(self._population)

    @property
    def function_names(self) -> List[str]:
        return list(self._name_index.keys())

    def get_metadata_field(self, field: str) -> Dict[Any, int]:
        stats = {}
        for individual in self._population.values():
            value = individual.metadata.get(field)
            stats[value] = stats.get(value, 0) + 1
        return stats

    def export_serializable(self) -> List[Dict]:
        return [
            {
                "id": indv.individual_id,
                "name": indv.func_name,
                "metadata": indv.metadata,
                "created_at": indv.created_at,
                "source_lines": len(indv.source_code.split('\n'))
            }
            for indv in self._population.values()
        ]
    
    @property
    def function_names(self) -> List[str]:
        return list(self._name_index.keys())

    def __iter__(self):
        return iter(self._population)
    
    def __getitem__(self, key):
        return self._population[key]

