"""
This script contains the screended Laplacian in dimension d on [0,1]^d.

This example corresponds to Example 1 in Section 3 of the manuscript.

"""

import tensorflow as tf
from tensorflow.keras.optimizers import Adam

from domains import UnitSquare
from integrators import RandomIntegrator, accuracy
from derivatives import grad_square
from models import create_model

# global 'hyperparameters' for this experiment
DIMENSION          = 2
ACTIVATION         = lambda x: tf.keras.activations.relu(x)**2
HIDDEN_LAYERS      = 4
NETWORK_WIDTH      = 64
ITERATIONS         = 10000
LEARNING_RATE      = 0.001
INTEGRATION_POINTS = 200000
OPTIMIZER          = Adam(learning_rate=LEARNING_RATE)


# computational domain
domain = UnitSquare(DIMENSION)

# random integrator using uniformly drawn points
integrator = RandomIntegrator(domain=domain, N=INTEGRATION_POINTS)

# accurate norm computation routine, used for evaluation
@accuracy(precision=1e-4, max_iter=5000)
def accurate_norm(func):
    base_integrator = RandomIntegrator(domain, 100000)
    return base_integrator(lambda x: func(x)**2)**(0.5)

# the un-normalized solution: u_tilde(x) = \sum_{i=1}^d cos(\pi x_i)
def u_tilde(x):
    return tf.reduce_sum(
        tf.math.cos(3.141592 * x), 
        axis=1,
        keepdims=True,
    )

# the solution, normalized version of u_tilde
norm = accurate_norm(u_tilde)
u_star = lambda x: 1./norm * u_tilde(x)

# the right-hand side f(x) = (\pi^2 + 1) * u_star(x) 
def f(x):
    return (3.141592**2 + 1) * u_star(x)

# the loss function, in this case the energy E
def loss_factory(integrator):
        @tf.function
        def DRM_loss(u_theta):
            E1 = 0.5 * integrator(lambda x: grad_square(u_theta, x))
            E2 = 0.5 * integrator(lambda x: u_theta(x)**2)
            E3 = -integrator(lambda x: u_theta(x) * f(x))
            return E1 + E2 + E3
        return DRM_loss

custom_loss = loss_factory(integrator=integrator)

# the model, we use a MLP with ReLU^2 activation
model = create_model(
    input_dim=DIMENSION,
    width=NETWORK_WIDTH,
    hidden_layers=HIDDEN_LAYERS,
    activation=ACTIVATION,
    )

# the error function, i.e., model - u_star
error = lambda x: model(x) - u_star(x)

# the training loop
prev_loss = 0.
for iteration in range(ITERATIONS):
    with tf.GradientTape() as tape:
        loss = custom_loss(model)
        
    gradients = tape.gradient(loss, model.trainable_variables)
    OPTIMIZER.apply_gradients(zip(gradients, model.trainable_variables))

    if iteration % 50 == 0:
        print(
                f'Iteration {iteration + 1} '
                f'Loss {loss} '
                f'Diff {prev_loss - loss} '
                f'Distance to Solution {accurate_norm(error)}'
            )
    prev_loss = loss






