import random
import numpy as np
import torchvision
import argparse
import inspect
import logging
import math
import os
from pathlib import Path
from typing import Optional
import datasets
import torch
import torch.nn.functional as F
from packaging import version
from torchvision import transforms
from tqdm.auto import tqdm
from sklearn.neighbors import LocalOutlierFactor
from torchvision.utils import make_grid, save_image
from sampler_main import *
from torchmetrics.image import FrechetInceptionDistance as FID
from roc_tpr import cal_roc_tpr
from dataset import *
import lpips
import architectures
import utils
import dnnlib
import pickle
# Will error if the minimal version of diffusers is not installed. Remove at your own risks.
# check_min_version("0.15.0.dev0")

#logger = get_logger(__name__, log_level="INFO")


def parse_args():
    parser = argparse.ArgumentParser(description="Test ood with consistency models")
    parser.add_argument('--seed',type=int, default=1557)
    args = parser.parse_args()


    return args


def load_architecture(name: str = "edm-ncsnpp-cifar10"):
    if name == "edm-ncsnpp-cifar10":
        arch = architectures.SongUNet(
            img_resolution=32,
            in_channels=3,
            out_channels=3,
            embedding_type="fourier",
            encoder_type="residual",
            decoder_type="standard",
            channel_mult_noise=2,
            resample_filter=[1,3,3,1],
            model_channels=128,
            channel_mult=[2,2,2],
            dropout=0.0,
        )

    else:
        raise ValueError(f"Unsupported architecture: {name}")

    return arch






def main(args):
    print(args)
    model = load_architecture().to('cuda')
    
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic=True
    np.random.seed(args.seed)
    random.seed(args.seed)
    init_pos = "/init_cm/diffusion_pytorch_model.bin" 
    pretrained_dict = torch.load(init_pos, map_location='cuda')
    src_dict = pretrained_dict

    npb_model = list(model.named_parameters()) + list(model.named_buffers())

    with torch.no_grad():
        for name, tensor in npb_model:
            assert name in src_dict
            tensor.copy_(src_dict[name])
    
            
    ##########################
    transform_test  = transforms.Compose([transforms.ToTensor(), transforms.Normalize((0.5,0.5,0.5),(0.5,0.5,0.5))])
    device = 'cuda'
    scheduler = utils.CMScheduler()

    N=18
    timesteps = scheduler.discretize_timesteps(N, device=device)

    # Get the dataset
    cifar10_train = torchvision.datasets.CIFAR10(root='./data', train=True, download=True, transform = transform_test)
    cifar10_test = torchvision.datasets.CIFAR10(root='./data', train=False, download=True, transform = transform_test)
    cifar100_test = torchvision.datasets.CIFAR100(root='./data', train=False, download=True, transform = transform_test)
    svhn_test = torchvision.datasets.SVHN(root='./data', split='test', download=True, transform = transform_test)
    lsun_test = torchvision.datasets.ImageFolder(root='./data/LSUN_pil', transform = transform_test)
    imgn_test = torchvision.datasets.ImageFolder(root='./data/Imagenet_pil', transform = transform_test)

    model.eval()

    cf10tr_loader = torch.utils.data.DataLoader(cifar10_train, batch_size= 1000)
    cf10te_loader = torch.utils.data.DataLoader(cifar10_test, batch_size = 1000)
    cf100te_loader = torch.utils.data.DataLoader(cifar100_test, batch_size = 1000)
    svhn_loader = torch.utils.data.DataLoader(svhn_test, batch_size=1000)
    lsun_loader = torch.utils.data.DataLoader(lsun_test, batch_size=1000)
    imgn_loader = torch.utils.data.DataLoader(imgn_test, batch_size=1000)
    


    print('cf10',len(cifar10_test))
    print('*'*100)
    cf10_recon = impaint_lpips(cf10te_loader,model,scheduler,timesteps,device) 
    print('cf100',len(cifar100_test))
    cf100_recon = impaint_lpips(cf100te_loader, model,scheduler,timesteps,device)
    print('svhn',len(svhn_test))
    svhn_recon = impaint_lpips(svhn_loader, model, scheduler, timesteps, device)
    print('lsun',len(lsun_test))
    lsun_recon = impaint_lpips(lsun_loader, model, scheduler, timesteps, device)
    print('imagenet',len(imgn_test))
    imgn_recon = impaint_lpips(imgn_loader, model, scheduler, timesteps, device)


    print(cal_roc_tpr(cf10_recon,cf100_recon,0.95))
    print(cal_roc_tpr(cf10_recon,svhn_recon,0.95))
    print(cal_roc_tpr(cf10_recon,lsun_recon,0.95))
    print(cal_roc_tpr(cf10_recon,imgn_recon,0.95))
        
    pref=str(args.seed)  
    os.makedirs('numpy_stats_LMD/cifar10/'+pref,exist_ok=True)
    shortcut = './numpy_stats_LMD/cifar10/'+pref
    np.savetxt(shortcut+'/'+'cifar10_test.txt',cf10_recon)
    np.savetxt(shortcut+'/'+'cifar100.txt',cf100_recon)
    np.savetxt(shortcut+'/'+'svhn.txt', svhn_recon)
    np.savetxt(shortcut+'/'+'lsun.txt',lsun_recon)
    np.savetxt(shortcut+'/'+'imgn.txt',imgn_recon)



if __name__ == "__main__":
    args = parse_args()
    main(args)


