#
# Copyright (C) 2023, Inria
# GRAPHDECO research group, https://team.inria.fr/graphdeco
# All rights reserved.
#
# This software is free for non-commercial, research and evaluation use 
# under the terms of the LICENSE.md file.
#
# For inquiries contact  george.drettakis@inria.fr
#

import os
os.environ['CUDA_LAUNCH_BLOCKING'] = '1'
import torch
import torch.nn.functional as F
from random import randint
from utils.loss_chamfer_utils import chamfer_distance
from utils.loss_utils import l1_loss, ce_loss, or_loss, ssim
from gaussian_renderer import render_hair, network_gui,render_hair_weight,render_hair_weight_fine,render_hair_weight_coarse,render_hair_weight_wo_active,render_hair_weight_sparse
import sys
import yaml
from scene import Scene, GaussianModel, GaussianModelHair, GaussRenderer
from utils.general_utils import safe_state
import uuid
from tqdm import tqdm
from utils.image_utils import psnr, vis_orient,psnr_mask
from argparse import ArgumentParser, Namespace
from arguments import ModelParams, PipelineParams, OptimizationParams, ModelHiddenParams, TextureHiddenParams
from torch.utils.data import DataLoader
import pickle as pkl
from utils.general_utils import build_rotation, DynamicFrameSampler
import time
from kaolin.metrics.pointcloud import sided_distance
import numpy as np
from collections import deque

try:
    from torch.utils.tensorboard import SummaryWriter
    TENSORBOARD_FOUND = True
except ImportError:
    TENSORBOARD_FOUND = False



def training(dataset, defor, texture_hidden, opt, opt_hair, pipe, testing_iterations, saving_iterations, checkpoint_iterations, model_path_hair, pointcloud_path_head, checkpoint, checkpoint_hair, debug_from):           
    first_iter = 0
    time_str = time.strftime("%Y-%m-%d-%H-%M-%S", time.localtime())
    print(f"log and checkpoint will be saved at {time_str}")
    # torch.multiprocessing.set_start_method('spawn',force=True)
    tb_writer = prepare_output_and_logger(dataset, model_path_hair, time_str)
    gaussians = GaussianModel(dataset.sh_degree)
    gaussians_hair = GaussianModelHair(dataset.source_path, dataset.flame_mesh_dir, opt_hair, defor, texture_hidden, dataset.sh_degree, dataset.start_time_step, dataset.num_time_steps)
    scene = Scene(dataset, gaussians, pointcloud_path=pointcloud_path_head, load_iteration=None)
    
    if opt_hair['general']['lr'] is not None:
        # opt_hair['general']['lr'] = 1e-8
        # opt_hair['general']['lr'] = 1e-3
        opt_hair['general']['lr'] = 1e-3
    gaussians.training_setup(opt)
    gaussians_hair.create_from_pcd(dataset.source_path, dataset.strand_scale, gaussians.spatial_lr_scale)
    gaussians_hair.training_setup(opt, opt_hair)
    # Pick a random Camera
    viewpoint_stack = None
    batch_size = opt.batch_size
    if not viewpoint_stack:
        viewpoint_stack = scene.getTrainCameras()
        torch.cuda.empty_cache()
    # viewpoint_cam = viewpoint_stack.pop(randint(0, len(viewpoint_stack)-1))
    sampler = DynamicFrameSampler(viewpoint_stack, loss_threshold = 0.050)   # total_loss
    cpu_count = os.cpu_count() - 1 or 4
    # print(f'Using {cpu_count} CPUs')
    viewpoint_stack_loader = DataLoader(viewpoint_stack, batch_size = batch_size, shuffle = False, num_workers = 16, sampler = sampler, collate_fn=list, pin_memory=True)
    # viewpoint_stack_loader = DataLoader(viewpoint_stack, batch_size = batch_size, shuffle = False, num_workers = 16, sampler = sampler, collate_fn=list, pin_memory=True)
    # viewpoint_stack_loader = DataLoader(viewpoint_stack, batch_size = batch_size, shuffle = False, num_workers = 0, sampler = sampler, collate_fn=list, pin_memory=True)
    # viewpoint_stack_loader = DataLoader(viewpoint_stack, batch_size = batch_size, shuffle = True, num_workers = 16, collate_fn = list, pin_memory = True)
    sampler.select_timestep(1,'True')
    # sampler.num_time_steps = 1
    # sampler.select_timestep(6,'False')
    # sampler.select_timestep(2,'False')
    loader = iter(viewpoint_stack_loader)
    if checkpoint:
        model_params, _ = torch.load(checkpoint)
        gaussians.restore(model_params, opt)
    if checkpoint_hair:
        print(f'Loading optimized hair {checkpoint_hair}')
        # model_params, first_iter = torch.load(checkpoint_hair)
        model_params, iteration_last = torch.load(checkpoint_hair)
        training_args = [opt,opt_hair]
        gaussians_hair.restore(model_params, training_args)   
        first_iter = iteration_last
    if dataset.trainable_cameras:
        print(f'Loading optimized cameras from iter {scene.loaded_iter}')
        params_cam_rotation, params_cam_translation, params_cam_fov = pkl.load(open(scene.model_path + "/cameras/" + str(scene.loaded_iter) + ".pkl", 'rb'))
        for k in scene.train_cameras.keys():
            for camera in scene.train_cameras[k]:
                if dataset.trainable_cameras:
                    camera._rotation_res.data = params_cam_rotation[camera.image_name]
                    camera._translation_res.data = params_cam_translation[camera.image_name]
                if dataset.trainable_intrinsics:
                    camera._fov_res.data = params_cam_fov[camera.image_name]

    with torch.no_grad():
        # Head gaussians data
        gaussians.mask_precomp = gaussians.get_label()[..., 0] < 0.6
        gaussians.xyz_precomp = gaussians.get_xyz()[gaussians.mask_precomp].detach()
        gaussians.opacity_precomp = gaussians.get_opacity()[gaussians.mask_precomp].detach()
        gaussians.scaling_precomp = gaussians.get_scaling()[gaussians.mask_precomp].detach()
        gaussians.rotation_precomp = gaussians.get_rotation()[gaussians.mask_precomp].detach()
        gaussians.cov3D_precomp = gaussians.get_covariance(1.0)[gaussians.mask_precomp].detach()
        gaussians.shs_view = gaussians.get_features()[gaussians.mask_precomp].detach().transpose(1, 2).view(-1, 3, (gaussians.max_sh_degree + 1)**2)
    
    # bg_color = [1, 1, 1, 0, 0, 0, 0, 0, 0, 100] if dataset.white_background else [0, 0, 0, 0, 0, 0, 0, 0, 0, 100]
    bg_color = [1, 1, 1, 0, 0, 0, 0, 0, 0, 0] if dataset.white_background else [0, 0, 0, 0, 0, 0, 0, 0, 0, 0]
    background = torch.tensor(bg_color, dtype=torch.float32, device="cuda")

    iter_start = torch.cuda.Event(enable_timing = True)
    iter_end = torch.cuda.Event(enable_timing = True)
  
    ema_loss_for_log = 0.0
    
    progress_bar = tqdm(range(first_iter, opt.iterations), desc="Training progress")
    first_iter += 1
    # coarse_iteration = 100000
    coarse_iteration = 10000
    grid_active_interval = 30
    # coarse_iteration = 0
    render_state = "coarse"
    state = "coarse" 
    update_state = False
    diff = 0
    sparse_iteration = 0
    # loss_list = deque(maxlen=500)
    loss_list = deque(maxlen=500)
    # gaussians_hair.deformaton_scale = 1e-1
    gaussians_hair.deformaton_scale = 1e-2
    # cam_id_list = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15]
    cam_id_list = [0,2,3,4,5,6,7,8,9,10,11,12,13,14,15]
    # cam_id_list = [0,1,3,4,5,6,7,8,9,10,11,12,13,14,15]
    # cam_id_list = [0,1,2,4,5,6,7,8,9,10,11,12,13,14,15]
    # cam_id_list = [0,1,2,3,5,6,7,8,9,10,11,12,13,14,15]
    # cam_id_list = [0,1,2,3,4,6,7,8,9,10,11,12,13,14,15]
    # cam_id_list = [0,1,2,3,4,5,7,8,9,10,11,12,13,14,15]
    # cam_id_list = [0,1,2,3,4,5,6,8,9,10,11,12,13,14,15]
    # cam_id_list = [0,1,2,3,4,5,6,7,9,10,11,12,13,14,15]
    # cam_id_list = [0,1,2,3,4,5,6,7,8,10,11,12,13,14,15]
    # cam_id_list = [0,1,2,3,4,5,6,7,8,9,11,12,13,14,15]
    # cam_id_list = [0,1,2,3,4,5,6,7,8,9,10,12,13,14,15]
    # cam_id_list = [0,1,2,3,4,5,6,7,8,9,10,11,13,14,15]
    # cam_id_list = [0,1,2,3,4,5,6,7,8,9,10,11,12,14,15]
    # cam_id_list = [0,1,2,3,4,5,6,7,8,9,10,11,12,13,15]
    # cam_id_list = [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14]
    viewpoint_indices = list(range(15))
    # gaussians_hair.deformaton_scale = 0
    viewpoint_cams = next(loader)
    viewpoint_cam = viewpoint_cams[0]
    # sparse_interval = 100
    # sparse_clamp = 90
    # coarse_steps = range(sparse_interval-sparse_clamp,sparse_interval)
    # sparse_steps = range(0,sparse_interval-sparse_clamp)
    for iteration in range(first_iter, opt.iterations + 1):
        if network_gui.conn == None:
            network_gui.try_connect()
        while network_gui.conn != None or False:
            try:
                net_image_bytes = None
                custom_cam, do_training, pipe.convert_SHs_python, pipe.compute_cov3D_python, keep_alive, scaling_modifer = network_gui.receive()
                if custom_cam != None:
                    net_image = render_hair(custom_cam, gaussians, gaussians_hair, pipe, background, render_state, scaling_modifer)["render"]
                    net_image_bytes = memoryview((torch.clamp(net_image, min=0, max=1.0) * 255).byte().permute(1, 2, 0).contiguous().cpu().numpy())
                network_gui.send(net_image_bytes, dataset.source_path)
                if do_training and ((iteration < int(opt.iterations)) or not keep_alive):
                    break
            except Exception as e:
                network_gui.conn = None
                
        if iteration == 1000:
            sampler.shuffle()

        iter_start.record()
        torch.cuda.synchronize()
        start_time = time.time()

        gaussians_hair.update_learning_rate(iteration, coarse_iteration)
        
        LCE = 0.0
        Ll1 = 0.0
        LOR = 0.0
        LDF = 0.0
        tv_loss = 0.0
        hair_smoothness = 0.0
        time_list = {}
        # import pdb;pdb.set_trace()
        # for viewpoint_cam in viewpoint_cams:
        # if not viewpoint_indices:
        #     viewpoint_indices = list(range(15))
        # rand_idx = randint(0, len(viewpoint_indices) - 1)
        # vind = viewpoint_indices.pop(rand_idx)
        # time_seq = (iteration - 1) % 15
        # viewpoint_cam = viewpoint_stack[time_seq*16 + cam_id_list[vind]]
        cam_id = viewpoint_cam.camera_index
        # viewpoint_cam = viewpoint_stack[(iteration-1) % 16]
        # cam_id = (iteration-1) % 16
        torch.cuda.synchronize()
        time2 = time.time()
        with torch.no_grad():
            # Head gaussians data
            # gaussians.mask_precomp = gaussians.get_label(viewpoint_cam.time_step)[..., 0] < 0.5
            gaussians_hair.training = True
            # gaussians.mask_precomp = gaussians.get_label(viewpoint_cam.time_step)[..., 0] < 0.6
            # gaussians.xyz_precomp = gaussians.get_xyz(viewpoint_cam.time_step)[gaussians.mask_precomp].detach()
            # gaussians.opacity_precomp = gaussians.get_opacity(viewpoint_cam.time_step)[gaussians.mask_precomp].detach()
            # gaussians.scaling_precomp = gaussians.get_scaling(viewpoint_cam.time_step)[gaussians.mask_precomp].detach()
            # gaussians.rotation_precomp = gaussians.get_rotation(viewpoint_cam.time_step)[gaussians.mask_precomp].detach()
            # gaussians.cov3D_precomp = gaussians.get_covariance(scaling_modifier = 1.0,time_index = viewpoint_cam.time_step)[gaussians.mask_precomp].detach()
            # gaussians.shs_view = gaussians.get_features(viewpoint_cam.time_step)[gaussians.mask_precomp].detach().transpose(1, 2).view(-1, 3, (gaussians.max_sh_degree + 1)**2)
        # gaussians_hair.initialize_gaussians_hair(iteration, num_strands = -1, time_step = viewpoint_cam.time_step)
        torch.cuda.synchronize()
        time3 = time.time()
        # if iteration > coarse_iteration and sampler.frames_to_load == 11:
        if iteration > coarse_iteration:
            sparse_iteration += 1 
            # if sparse_iteration % 100 == 0 or sparse_iteration == 1:
            if sparse_iteration % grid_active_interval == 0:
                # gaussians_hair.initialize_gaussians_hair(iteration, num_strands = 50000, viewpoint_cam = viewpoint_cam, state="sparse")
                gaussians_hair.idx_active_mask = None
                gaussians_hair.idx_hair_active_mask = None
                gaussians_hair.points_mask_active_hair_indices = None
                gaussians_hair.points_mask_wo_active_hair_indices = None
                state = "coarse" 
                gaussians_hair.initialize_gaussians_hair(iteration, num_strands = -1, viewpoint_cam = viewpoint_cam, state=state)
                render_pkg = render_hair_weight_coarse(viewpoint_cam, gaussians, gaussians_hair, pipe, background, render_state)
            else:
                state = "sparse" 
                # state = "sparse" 
                gaussians_hair.initialize_gaussians_hair(iteration, num_strands = -1, viewpoint_cam = viewpoint_cam, state=state)
                render_pkg = render_hair_weight_coarse(viewpoint_cam, gaussians, gaussians_hair, pipe, background, render_state)
                # render_pkg = render_hair_weight_sparse(viewpoint_cam, gaussians, gaussians_hair, pipe, background, render_state)
        else:
            gaussians_hair.initialize_gaussians_hair(iteration, num_strands = -1, viewpoint_cam = viewpoint_cam, state="coarse")
            render_pkg = render_hair_weight_coarse(viewpoint_cam, gaussians, gaussians_hair, pipe, background, render_state)
        # gaussians_hair.initialize_gaussians_hair(iteration, time_step = viewpoint_cam.time_step, state="coarse")
        # print(gaussians_hair._xyz.shape)
        # Render
        if (iteration - 1) == debug_from:
            pipe.debug = True
        torch.cuda.synchronize()
        time4 = time.time()
        # render_pkg = render_hair(viewpoint_cam, gaussians, gaussians_hair, pipe, background, render_state)
        # render_pkg = render_hair_weight_coarse(viewpoint_cam, gaussians, gaussians_hair, pipe, background, render_state)
        # render_pkg = render_hair_weight_fine(viewpoint_cam, gaussians, gaussians_hair, pipe, background, render_state)
        # render_pkg = render_hair_torch(gaussians_hair.gaussRender, viewpoint_cam, gaussians, gaussians_hair, pipe, background, render_state)
        # render_pkg = render_hair_torch_partial(gaussians_hair.gaussRender, viewpoint_cam, gaussians, gaussians_hair, pipe, background, render_state)
        torch.cuda.synchronize()
        time5 = time.time()
        image = render_pkg["render"]
        mask = render_pkg["mask"] 
        # mask = render_pkg["mask"]
        orient_angle = render_pkg["orient_angle"]
        orient_conf = render_pkg["orient_conf"]
        screenspace_points = render_pkg["viewspace_points"]
        # visibility_filter = render_pkg["visibility_filter"]
        # radii = render_pkg["radii"]
        # import ipdb;ipdb.set_trace()
        # Loss
        gt_image = viewpoint_cam.original_image # 3 x height x width
        gt_mask = viewpoint_cam.original_mask # 2 x height x width
        # gt_mask = (gt_mask > 0.0).int()
        # gt_mask_face = viewpoint_cam.original_mask_face
        gt_orient_angle = viewpoint_cam.original_orient_angle # 1 x height x width
        gt_orient_conf = viewpoint_cam.original_orient_conf # 1 x height x width

        
        # if gaussians_hair.image_ROI_mask.shape[0] != 0 and sparse_iteration > 0 and state == "sparse":

        #     gt_image = gt_image * gaussians_hair.image_ROI_mask
        #     intersect = (mask[:1] > 0) & (gt_mask[:1] > 0)
        #     gt_mask[:1] = gt_mask[:1] * intersect
        #     gt_orient_angle = gt_orient_angle * gaussians_hair.image_ROI_mask
        #     gt_orient_conf = gt_orient_conf * gaussians_hair.image_ROI_mask
        #     image = image * gaussians_hair.image_ROI_mask
        #     orient_angle = orient_angle * gaussians_hair.image_ROI_mask
        #     orient_conf = orient_conf * gaussians_hair.image_ROI_mask

        LCE += l1_loss(mask[:1], gt_mask[:1]) #, mask=(gt_mask[:1] < 0.5).detach())
        Ll1 += l1_loss(image, gt_image)
    
        # These losses are only applied for hair
        orient_weight = torch.ones_like(gt_mask[:1])
        if opt.use_gt_orient_conf: orient_weight = orient_weight * gt_orient_conf
        if not opt.train_orient_conf: orient_conf = None
        # LOR += or_loss(orient_angle, gt_orient_angle, orient_conf, weight=orient_weight, mask=gt_mask[:1])
        # if gaussians_hair.image_ROI_mask.shape[0] != 0 and sparse_iteration > 0 and state == "sparse":
        #     LOR += or_loss(orient_angle, gt_orient_angle, orient_conf, weight=orient_weight, mask=gaussians_hair.image_ROI_mask)
        # else:
        #     LOR += or_loss(orient_angle, gt_orient_angle, orient_conf, weight=orient_weight, mask=gt_mask[:1])
        LOR += or_loss(orient_angle, gt_orient_angle, orient_conf, weight=orient_weight, mask=gt_mask[:1])
        # tv_loss += gaussians_hair.compute_regulation(defor.time_smoothness_weight, defor.l1_time_planes, defor.plane_tv_weight)
        # Diffusion loss
        LDF += gaussians_hair.LDiff if gaussians_hair.LDiff is not None else torch.zeros_like(LOR)
        hair_smoothness += gaussians_hair.hair_smoothness if gaussians_hair.hair_smoothness is not None else torch.zeros_like(Ll1)

        if torch.isnan(Ll1).any(): Ll1 = torch.zeros_like(Ll1)
        if torch.isnan(LCE).any(): LCE = torch.zeros_like(Ll1)
        if torch.isnan(LOR).any(): LOR = torch.zeros_like(Ll1)
        if torch.isnan(LDF).any(): LDF = torch.zeros_like(Ll1)
            
        loss = (
            Ll1 * opt.lambda_dl1 + 
            LCE * opt.lambda_dmask +
            LOR * opt.lambda_dorient + 
            LDF * opt.lambda_dsds +
            # tv_loss * opt.lambda_tv+
            hair_smoothness * opt.lambda_hair_smoothness
        )
        loss_list.append(loss.item())
        if len(loss_list) == loss_list.maxlen:
            Q1 = np.percentile(loss_list, 25)
            Q3 = np.percentile(loss_list, 75)
            IQR = Q3 - Q1
            lower_bound = Q1 - 1.5 * IQR
            upper_bound = Q3 + 1.5 * IQR
            loss_list_filter = [x for x in loss_list if lower_bound <= x <= upper_bound]
            diff = max(loss_list_filter)-min(loss_list_filter)
            if diff <= 0.045 and iteration % 50 == 0:
                update_state = True
                
        loss.backward()
        torch.cuda.synchronize()
        time6 = time.time()
        
        try:
            viewpoint_cams = next(loader)
        except StopIteration:
            loader = iter(viewpoint_stack_loader)
            sampler.shuffle()
            viewpoint_cams = next(loader)
        viewpoint_cam = viewpoint_cams[0]
        torch.cuda.synchronize()
        time7 = time.time()
        # if sparse_iteration > 0 and (sparse_iteration % 10 == 0 or sparse_iteration == 1):
        # if sparse_iteration > 0 and (state == "sparse" or sparse_iteration % 100 == coarse_steps[-1] or sparse_iteration == 1):
        #     if gaussians_hair._pts.shape[0] > 0:   
        #         gaussian_hair_grad = screenspace_points.grad[gaussians.mask_precomp.shape[0]:]
        #         gaussians_hair.gaussian_hair_grad = gaussian_hair_grad
        #     else:
        #         gaussian_hair_grad = gaussians_hair.gaussian_hair_grad
        #     # print("gaussian_hair_grad.shape:  ",gaussian_hair_grad.shape)
        #     if gaussian_hair_grad.shape[0] != 0:
        #         gaussians_hair.densify_and_prune_monocular(gaussian_hair_grad, viewpoint_cam, num_threshold = 20000, sparse_iter = sparse_iteration, coarse_iteration = coarse_iteration, 
        #                                                 densification_interval = 10, densification_recover_interval = 2000, prune_threshold = 3 * 1e-5, densification_threshold = 5 * 1e-4)
        # if sparse_iteration > 0 and (state == "sparse" or sparse_iteration % 100 == coarse_steps[-1] or sparse_iteration == 1):
        # if iteration > 0 and iteration < coarse_iteration and iteration % 50 == 0:
        #     gaussian_hair_grad = screenspace_points.grad[gaussians.mask_precomp.shape[0]:]
        #     if gaussian_hair_grad.shape[0] != 0:
        #         gaussians_hair.densify_and_prune(gaussian_hair_grad, num_threshold = 10000, iteration=iteration, densification_threshold = 1 * 1e-4)
        if iteration == coarse_iteration:
            gaussians_hair.strands_generator.int_para_precomp(0)    
        if iteration >= coarse_iteration and iteration < coarse_iteration + 16:
            gaussians_hair.add_densification_stats_average(cam_id)
        if iteration >= coarse_iteration + 16 and sparse_iteration > 0 and sparse_iteration % grid_active_interval == 0:
        # if iteration >= coarse_iteration + 16 and sparse_iteration > 0:
            gaussians_hair.add_densification_stats_average(cam_id)
            gaussian_head_grad = screenspace_points.grad[:gaussians.mask_precomp.shape[0]]
            gaussians_hair.update_grid_active_set(num_threshold = 10000, iteration = iteration, sparse_iteration = sparse_iteration, 
                                             coarse_iteration = coarse_iteration, grid_active_interval = grid_active_interval, grid_active_threshold = 1e-4)
        if iteration > 0 and iteration < coarse_iteration and iteration % gaussians_hair.grid_interval == 0:
            gaussians_hair.densify_and_prune_v2(num_threshold = 10000, iteration=iteration, densification_threshold = 1 * 1e-4)
            # gaussians_hair.update_head_active_set(gaussian_head_grad, grid_active_threshold = 1e-6)
        torch.cuda.synchronize()
        time8 = time.time()
        # if sparse_iteration == 1:
        #     gaussians_hair.reset_optimizers(opt, opt_hair,decay=1e-1)
        # Optimizer step
        if iteration < opt.iterations:
            for param in gaussians_hair.optimizer_strands.param_groups[0]['params']:
                if param.grad is not None and param.grad.isnan().any():
                    gaussians_hair.optimizer_strands.zero_grad()
                    print('NaN during backprop was found, skipping iteration...')
            # for param in gaussians_hair.optimizer_strands_grid.param_groups[0]['params']:
            #     if param.grad is not None and param.grad.isnan().any():
            #         gaussians_hair.optimizer_strands_grid.zero_grad()
            #         print('NaN during backprop was found, skipping iteration...')
            for param in gaussians_hair.optimizer_gaussianWeight.param_groups[0]['params']:
                if param.grad is not None and param.grad.isnan().any():
                    gaussians_hair.optimizer_gaussianWeight.zero_grad()
                    print('NaN during backprop was found, skipping iteration...')
            # for param in gaussians_hair.optimizer_deformation.param_groups[0]['params']:
            #     if param.grad is not None and param.grad.isnan().any():
            #         gaussians_hair.optimizer_deformation.zero_grad()
            #         print('NaN during backprop was found, skipping iteration...')
            if iteration >= coarse_iteration + 16 and sparse_iteration > 0 and sparse_iteration % grid_active_interval == 0 and False:
                gaussians_hair.optimizer_strands.zero_grad(set_to_none = True)
                gaussians_hair.optimizer_strands_grid.zero_grad(set_to_none = True)
                gaussians_hair.optimizer_gaussianWeight.zero_grad(set_to_none = True)
            else:
                gaussians_hair.optimizer_strands.step()
                gaussians_hair.optimizer_strands.zero_grad(set_to_none = True)
                gaussians_hair.optimizer_strands_grid.step()
                gaussians_hair.optimizer_strands_grid.zero_grad(set_to_none = True)
                gaussians_hair.optimizer_gaussianWeight.step()
                gaussians_hair.optimizer_gaussianWeight.zero_grad(set_to_none = True)
            # gaussians_hair.optimizer_strands.step()
            # gaussians_hair.optimizer_strands.zero_grad(set_to_none = True)
            # gaussians_hair.optimizer_strands_grid.step()
            # gaussians_hair.optimizer_strands_grid.zero_grad(set_to_none = True)
            # gaussians_hair.optimizer_gaussianWeight.step()
            # gaussians_hair.optimizer_gaussianWeight.zero_grad(set_to_none = True)
        # if iteration >= coarse_iteration and sparse_iteration > 0 and sparse_iteration % grid_active_interval == 0:
        #     gaussians_hair.update_wo_active_set_data(scene, render_hair_weight_wo_active,(pipe, background,render_state))
        # if iteration == coarse_iteration + 16:
        #     for param_group in gaussians_hair.optimizer_strands.param_groups:
        #          param_group['lr'] *= 0.5
        #     for param_group in gaussians_hair.optimizer_strands_grid.param_groups:
        #         param_group['lr'] *= 0.5
                
        iter_end.record()
        torch.cuda.synchronize()
        end_time = time.time()
        
        time_list["D_L"] = time2 - start_time
        time_list["G_P"] = time3 - time2
        time_list["GH_P"] = time4 - time3
        time_list["Render"] = time5 - time4
        time_list["L_B"] = time6 - time5
        time_list["data_loader"] = time7 - time6
        time_list["update"] = time8 - time7
        time_list["Opt"] = end_time - time8

        with torch.no_grad():
            # Progress bar
            ema_loss_for_log = 0.4 * loss.item() + 0.6 * ema_loss_for_log
            if iteration % 10 == 0:
                progress_bar.set_postfix({"Loss": f"{ema_loss_for_log:.{7}f}"})
                progress_bar.update(10)
            if iteration == opt.iterations:
                progress_bar.close()

            # Log and save
            training_report(tb_writer, iteration, 16, diff, gaussians_hair.loss_pts, Ll1, LCE, LOR, LDF, hair_smoothness, loss, l1_loss, ce_loss, or_loss, iter_start.elapsed_time(iter_end), end_time-start_time, time_list, testing_iterations, state, scene, gaussians_hair, render_hair, (pipe, background,None))

            # if (iteration in checkpoint_iterations):
            if (iteration % 1000 == 0):
                print("\n[ITER {}] Saving Checkpoint".format(iteration))
                os.makedirs(model_path_hair + "/checkpoints/"+time_str, exist_ok=True)
                torch.save((gaussians_hair.capture(), iteration), model_path_hair + f"/checkpoints/{time_str}/" + str(iteration) + ".pth")
                # torch.save(gaussians_hair.strands_generator.texture_decoder.state_dict(), f'{model_path_hair}/checkpoints/texture_decoder.pth')

def prepare_output_and_logger(args, model_path_hair, time_str):    
    if not model_path_hair:
        if os.getenv('OAR_JOB_ID'):
            unique_str=os.getenv('OAR_JOB_ID')
        else:
            unique_str = str(uuid.uuid4())
        model_path_hair = os.path.join("./output/", unique_str[0:10])
        
    # Set up output folder
    print("Output folder: {}".format(model_path_hair))
    os.makedirs(model_path_hair, exist_ok = True)
    with open(os.path.join(model_path_hair, "cfg_args"), 'w') as cfg_log_f:
        cfg_log_f.write(str(Namespace(**vars(args))))

    # time_str = time.strftime("%Y-%m-%d-%H-%M-%S", time.localtime())
    tb_writer = None
    # Create Tensorboard writer
    tb_writer = None
    if TENSORBOARD_FOUND:
        # tb_writer = SummaryWriter(model_path_hair)
        tb_writer = SummaryWriter(os.path.join(model_path_hair, "logs",time_str))
    else:
        print("Tensorboard not available: not logging progress")
    return tb_writer

def training_report(tb_writer, iteration, len_loader, diff ,loss_pts, Ll1, LCE, LOR, LDF, hair_smoothness, loss, l1_loss, ce_loss, or_loss, GPU_elapsed, elapsed, time_list, testing_iterations, state, scene : Scene, gaussians_hair, renderFunc, renderArgs):
    gaussians_hair.time_used += elapsed
    gaussians_hair.GPU_time_used += GPU_elapsed
    time_total = time_list["D_L"] + time_list["G_P"] + time_list["GH_P"] + time_list["Render"] + time_list["L_B"] + time_list["Opt"]
    if tb_writer:
        tb_writer.add_scalar('train_loss_patches/uvs_count', gaussians_hair.uvs_mask_count, iteration)
        tb_writer.add_scalar('train_loss_patches/uvs_active_count', gaussians_hair.uvs_mask_active_count, iteration)
        tb_writer.add_scalar('train_loss_patches/hair_active_nums', gaussians_hair.hair_active_num, iteration)
        tb_writer.add_scalar('train_loss_patches/Loader', len_loader, iteration)
        tb_writer.add_scalar('train_loss_patches/diff', diff, iteration)
        tb_writer.add_scalar('train_loss_patches/loss_pts', loss_pts, iteration)
        tb_writer.add_scalar('train_loss_patches/hair_smoothness', hair_smoothness, iteration)
        tb_writer.add_scalar('train_loss_patches/l1_loss', Ll1.item(), iteration)
        tb_writer.add_scalar('train_loss_patches/ce_loss', LCE.item(), iteration)
        tb_writer.add_scalar('train_loss_patches/or_loss', LOR.item(), iteration)
        tb_writer.add_scalar('train_loss_patches/df_loss', LDF.item(), iteration)
        # tb_writer.add_scalar('train_loss_patches/uvs_index_select', gaussians_hair.strands_generator.uvs_index_select.shape[0], iteration)
        tb_writer.add_scalar('train_loss_patches/total_loss', loss.item(), iteration)
        tb_writer.add_scalar('time/iter_gpu_time', GPU_elapsed, iteration)
        tb_writer.add_scalar('time/iter_time', elapsed, iteration)
        # tb_writer.add_scalar('time/total_gpu_time',gaussians_hair.GPU_time_used/1000.00, iteration)
        # tb_writer.add_scalar('time/total_time',gaussians_hair.time_used, iteration)
        # tb_writer.add_scalar('time/average_gpu_time',(gaussians_hair.GPU_time_used/1000.00)/iteration, iteration)
        # tb_writer.add_scalar('time/average_time',gaussians_hair.time_used/iteration, iteration)
        tb_writer.add_scalar('time/average_gpu_time(h/w)',(gaussians_hair.GPU_time_used/1000.00)/iteration*10000/3600, iteration)
        tb_writer.add_scalar('time/average_time(h/w)',gaussians_hair.time_used/iteration*10000/3600, iteration)
        tb_writer.add_scalar('time/data_preprocess',time_list["D_L"], iteration)
        tb_writer.add_scalar('time/gaussian_preprocess',time_list["G_P"], iteration)
        tb_writer.add_scalar('time/gaussian_hair_preprocess',time_list["GH_P"], iteration)
        tb_writer.add_scalar('time/render',time_list["Render"], iteration)
        tb_writer.add_scalar('time/loss_back',time_list["L_B"], iteration)
        tb_writer.add_scalar('time/data_loader',time_list["data_loader"], iteration)
        tb_writer.add_scalar('time/gradient_update',time_list["update"], iteration)
        tb_writer.add_scalar('time/optimizer',time_list["Opt"], iteration)
        tb_writer.add_scalar("scaling/0", gaussians_hair.get_scaling[:, 0].mean().item(), iteration)
        tb_writer.add_scalar("scaling/1", gaussians_hair.get_scaling[:, 1].mean(), iteration)
        tb_writer.add_scalar("scaling/2", gaussians_hair.get_scaling[:, 2].mean(), iteration)

    # Report test and samples of training set
    if iteration % 2000 == 0:
        validation_configs = [{'name': 'train', 'cameras' : [scene.getTrainCameras()[idx % len(scene.getTrainCameras())] for idx in range(0, 2500, 219)]},
                              {'name': 'test', 'cameras' : [scene.getTestCameras()[idx % len(scene.getTestCameras())] for idx in range(0, 145, 20)]}]

        for config in validation_configs:
            if config['cameras'] and len(config['cameras']) > 0:
                l1_test = 0.0
                ce_test = 0.0
                or_test = 0.0
                psnr_test = 0.0
                psnr_test_mask = 0.0
                for idx, viewpoint in enumerate(config['cameras']):
                    torch.cuda.empty_cache()
                    gaussians_hair.training = False
                    gaussians_hair.initialize_gaussians_hair(-1, viewpoint_cam = viewpoint, state = "coarse")
                    # render_pkg = renderFunc(viewpoint, scene.gaussians, gaussians_hair, *renderArgs)
                    render_pkg = render_hair_weight_coarse(viewpoint, scene.gaussians, gaussians_hair, *renderArgs)
                    gt_image = torch.clamp(viewpoint.original_image.to("cuda"), 0.0, 1.0)
                    gt_mask = torch.clamp(viewpoint.original_mask.to("cuda"), 0.0, 1.0)
                    gt_mask_face = torch.clamp(viewpoint.original_mask_face.to("cuda"), 0.0, 1.0)
                    # image = torch.clamp(render_pkg["render"] * torch.any(gt_mask>0, dim=0,keepdim=True), 0.0, 1.0)
                    image = torch.clamp(render_pkg["render"], 0.0, 1.0)
                    # mask = torch.clamp(render_pkg["mask"] * torch.any(image>0, dim=0,keepdim=True), 0.0, 1.0)
                    # orient_angle = torch.clamp(render_pkg["orient_angle"] * torch.any(image>0, dim=0,keepdim=True), 0.0, 1.0)
                    mask = torch.clamp(render_pkg["mask"], 0.0, 1.0)
                    orient_angle = torch.clamp(render_pkg["orient_angle"], 0.0, 1.0)
                    orient_conf = render_pkg["orient_conf"]
                    orient_conf_vis = (1 - 1 / (orient_conf + 1)) * mask[:1]
                    
                    gt_orient_angle = torch.clamp(viewpoint.original_orient_angle.to("cuda"), 0.0, 1.0)
                    gt_orient_conf = viewpoint.original_orient_conf.to("cuda")
                    gt_orient_conf_vis = (1 - 1 / (gt_orient_conf + 1)) * gt_mask[:1]
                    # if tb_writer and (idx < 5):
                    if tb_writer:
                        tb_writer.add_images(config['name'] + f"_cam_{viewpoint.camera_id}_time_{viewpoint.image_name}/render", image[None], global_step=iteration)
                        tb_writer.add_images(config['name'] + f"_cam_{viewpoint.camera_id}_time_{viewpoint.image_name}/render_gt", torch.abs(gt_image[None] - image[None]), global_step=iteration)
                        tb_writer.add_images(config['name'] + f"_cam_{viewpoint.camera_id}_time_{viewpoint.image_name}/render_mask", F.pad(mask, (0, 0, 0, 0, 0, 3-mask.shape[0]), 'constant', 0)[None], global_step=iteration)
                        tb_writer.add_images(config['name'] + f"_cam_{viewpoint.camera_id}_time_{viewpoint.image_name}/render_orient", vis_orient(orient_angle, mask[:1])[None], global_step=iteration)
                        tb_writer.add_images(config['name'] + f"_cam_{viewpoint.camera_id}_time_{viewpoint.image_name}/render_orient_conf", vis_orient(orient_angle, orient_conf_vis)[None], global_step=iteration)
                    if (iteration == 10  or iteration == 200 or iteration == 10000 or iteration == 40200 or iteration == 1000) and tb_writer:
                        tb_writer.add_images(config['name'] + f"_cam_{viewpoint.camera_id}_time_{viewpoint.image_name}/ground_truth", gt_image[None], global_step=iteration)
                        tb_writer.add_images(config['name'] + f"_cam_{viewpoint.camera_id}_time_{viewpoint.image_name}/ground_truth_mask", F.pad(gt_mask, (0, 0, 0, 0, 0, 3-gt_mask.shape[0]), 'constant', 0)[None], global_step=iteration)
                        tb_writer.add_images(config['name'] + f"_cam_{viewpoint.camera_id}_time_{viewpoint.image_name}/ground_truth_orient", vis_orient(gt_orient_angle, gt_mask[:1])[None], global_step=iteration)
                        tb_writer.add_images(config['name'] + f"_cam_{viewpoint.camera_id}_time_{viewpoint.image_name}/ground_truth_orient_conf", vis_orient(gt_orient_angle, gt_orient_conf_vis)[None], global_step=iteration)
                    l1_test += l1_loss(image * gt_mask[:1], gt_image * gt_mask[:1]).mean().double()
                    ce_test += ce_loss(mask[:1], gt_mask[:1]).mean().double()
                    or_test += or_loss(orient_angle, gt_orient_angle, mask=gt_mask[:1], weight=gt_orient_conf).mean().double()
                    psnr_test += psnr(image * gt_mask[:1], gt_image * gt_mask[:1]).mean().double()
                    psnr_test_mask += psnr_mask(gt_image, image, gt_mask[:1]).mean().double()
                    torch.cuda.empty_cache()
                psnr_test /= len(config['cameras'])
                l1_test /= len(config['cameras'])          
                ce_test /= len(config['cameras'])
                or_test /= len(config['cameras'])
                psnr_test_mask /= len(config['cameras'])
                print("\n[ITER {}] Evaluating {}: L1 {} CE {} OR {} PSNR {}".format(iteration, config['name'], l1_test, ce_test, or_test, psnr_test))
                if tb_writer:
                    tb_writer.add_scalar(config['name'] + '/loss_viewpoint - l1_loss', l1_test, iteration)
                    tb_writer.add_scalar(config['name'] + '/loss_viewpoint - ce_loss', ce_test, iteration)
                    tb_writer.add_scalar(config['name'] + '/loss_viewpoint - or_loss', or_test, iteration)
                    tb_writer.add_scalar(config['name'] + '/loss_viewpoint - psnr', psnr_test, iteration)
                    tb_writer.add_scalar(config['name'] + '/metrics_PSNR',psnr_test_mask,iteration)
                    tb_writer.add_scalar(config['name'] + '/metrics_Time',gaussians_hair.time_used, iteration)

        torch.cuda.empty_cache()

if __name__ == "__main__":
    # Set up command line argument parser
    parser = ArgumentParser(description="Training script parameters")
    lp = ModelParams(parser)
    op = OptimizationParams(parser)
    pp = PipelineParams(parser)
    hp = ModelHiddenParams(parser)
    tp = TextureHiddenParams(parser)
    parser.add_argument('--ip', type=str, default="127.0.0.1")
    parser.add_argument('--port', type=int, default=6009)
    parser.add_argument('--debug_from', type=int, default=-1)
    parser.add_argument('--detect_anomaly', action='store_true', default=False)
    parser.add_argument("--test_iterations", nargs="+", type=int, default=[1_000, 5_000, 10_000, 15_000, 20_000])
    parser.add_argument("--save_iterations", nargs="+", type=int, default=[1_000, 5_000, 10_000, 15_000, 20_000])
    parser.add_argument("--quiet", action="store_true")
    parser.add_argument("--checkpoint_iterations", nargs="+", type=int, default=[1_000, 5_000, 10_000, 15_000, 20_000])
    parser.add_argument("--start_checkpoint", type=str, default = None)
    parser.add_argument("--start_checkpoint_hair", type=str, default = None)
    parser.add_argument("--hair_conf_path", type=str, default = None)
    parser.add_argument("--model_path_hair", type=str, default = None)
    parser.add_argument("--pointcloud_path_head", type=str, default = None)
    parser.add_argument("--configs", type=str, default = "")
    # parser.add_argument("--only_first_frame", action='store_true', default=False)
    args = parser.parse_args(sys.argv[1:])
    args.save_iterations.append(args.iterations)
    
    print("Optimizing " + args.model_path_hair)
    iterations = args.iterations
    # Configuration of hair strands
    with open(args.hair_conf_path, 'r') as f:
        replaced_conf = str(yaml.load(f, Loader=yaml.Loader)).replace('DATASET_TYPE', 'monocular')
        opt_hair = yaml.load(replaced_conf, Loader=yaml.Loader)

    if args.configs:
        import mmcv
        from utils.params_utils import merge_hparams
        config = mmcv.Config.fromfile(args.configs)
        args = merge_hparams(args, config)
    
    # Initialize system state (RNG)
    safe_state(args.quiet)
    args.iterations = iterations
    # Start GUI server, configure and run training
    network_gui.init(args.ip, args.port)
    torch.autograd.set_detect_anomaly(args.detect_anomaly)
    training(lp.extract(args), hp.extract(args), tp.extract(args), op.extract(args), opt_hair, pp.extract(args), args.test_iterations, args.save_iterations, args.checkpoint_iterations, args.model_path_hair, args.pointcloud_path_head, args.start_checkpoint, args.start_checkpoint_hair, args.debug_from)

    # All done
    print("\nTraining complete.")