module PolynomialTools

using DynamicPolynomials

import StaticPolynomials: PolynomialSystem, evaluate

export polynomial_system, polynomial_systems

"""
    polynomial_system(p[, v])

Create a polynomial system from `p` consisting of variables `v`.

If `v` is not specified, it is constructed from the variables of `p`.

See also [`polynomial_systems`](@ref).

# Examples
```jldoctest
using DynamicPolynomials
using PolynomialTools

@polyvar x y z

ps = polynomial_system([x^2, 2 * y * z])
ps([1, 1, 1])

# output

2-element Vector{Float64}:
 1.0
 2.0
```
"""
function polynomial_system(p, v)     
  polynomial_system(polynomial.(p), v)
end

function polynomial_system(p)
  polynomial_system(p, variables(p))
end

function polynomial_system(p::AbstractVector{<: AbstractPolynomial}, v)
  v_idx = [findfirst(==(vᵢ), v) for vᵢ ∈ variables(p)]
  n = length(p)
  p_nz_idx = p .≠ 0
  if any(p_nz_idx)
    ps = PolynomialSystem(p[p_nz_idx])
    x -> begin
      p_eval = zeros(n)
      p_eval[p_nz_idx] .= evaluate(ps, x[v_idx])
      p_eval
    end
  else
    _ -> zeros(n)
  end
end

"""
    polynomial_systems(p_vec[, v])

Create a vector of polynomial systems from the polynomial collections in `p_vec`.

The created systems share the same variables, `v`. If `v` is not specified, it is constructed from the variables of the polynomial collections in `p_vec`.

See also [`polynomial_system`](@ref).

# Examples
```jldoctest
using DynamicPolynomials
using PolynomialTools

@polyvar x y z

p₁ = [x^2, 2 * y * z]
p₂ = [2 * x * y]

ps = polynomial_systems([p₁, p₂])
ps[1]([1, 1, 1]), ps[2]([1, 1, 1])

# output

([1.0, 2.0], [2.0])
```
"""
function polynomial_systems(p_vec, v)
  [polynomial_system(pᵢ, v) for pᵢ ∈ p_vec]
end

function polynomial_systems(p_vec)
  polynomial_systems(p_vec, variables(reduce(vcat, p_vec)))
end

# DOCME
function DynamicPolynomials.variables(
  p_vec::AbstractVector{<: AbstractVector{<: AbstractPolynomialLike}}
)
  variables(reduce(vcat, p_vec))
end

end # module PolynomialTools
