module SINDyMethod

using DynamicPolynomials
using JuMP

import SCS: SCS

import ...DEFAULT_ZERO_THRESHOLD, ...DEFAULT_ABSOLUTE_TOLERANCE
import ...rescale_data!

export SINDy

# DOCME
function SINDy(
  x_data,
  ẋ_data,
  b; 
  SOCP_optimizer=optimizer_with_attributes(
    SCS.Optimizer, 
    "eps_abs" => DEFAULT_ABSOLUTE_TOLERANCE,
    "eps_rel" => 0.0,
    "max_iters" => 1e6
  ),
  zero_threshold=DEFAULT_ZERO_THRESHOLD, 
  η=1e-4,
  rescale=true,
  verbose=false
)

  Θ = [bᵢ(x) for x ∈ x_data, bᵢ ∈ b]

  if rescale
    scaling = rescale_data!(Θ)
  end

  model = Model(SOCP_optimizer)
  if !verbose
    set_silent(model)
  end
  @variable(model, ξ[eachindex(b)])
  @variable(model, ϵ_L₂_norm)

  # If the LASSO parameter is non-positive we simplify the models.
  if η > 0
    @variable(model, ξ_L₁_norm)
    @constraint(model, [ξ_L₁_norm; ξ] in MOI.NormOneCone(1 + length(ξ)))
    @objective(model, Min, ϵ_L₂_norm + η * ξ_L₁_norm)
  else
    @objective(model, Min, ϵ_L₂_norm)
  end
  y = Θ * ξ
  p_values = Polynomial[]

  for i ∈ eachindex(first(ẋ_data))

    ẋ_dataᵢ = [ẋ[i] for ẋ ∈ ẋ_data]

    if rescale
      scalingᵢ = scaling / only(rescale_data!(ẋ_dataᵢ))
      zero_thersholdᵢ = zero_threshold * scalingᵢ
    else
      zero_thersholdᵢ = zero_threshold
    end

    # Impose and keep track of any constrraints exclusive to the current model.
    ϵ_L₂_norm_constraint = @constraint(model, 
      [ϵ_L₂_norm; y - ẋ_dataᵢ] ∈ MOI.SecondOrderCone(1 + length(y))
    ) 

    # Optimize the current model.
    optimize!(model)
    status = termination_status(model)
    if status ≠ MOI.OPTIMAL
      @warn "SINDy terminated with status $(repr(status))"
    end
    ξ_value = value.(ξ)
    ξ_value[- zero_thersholdᵢ .< ξ_value .< zero_thersholdᵢ] .= 0

    if rescale
      ξ_value ./= scalingᵢ
    end

    p_value = polynomial(ξ_value , b)
    @info "SINDy found a polynomial" p=p_value
    push!(p_values, p_value)

    # Remove any constraints that should imposed only in the current model.
    delete(model, ϵ_L₂_norm_constraint)

  end

  p_values

end

function SINDy(x_data, w_data, ẋ_data, b; kwargs...)
  SINDy(
    [vcat(x, w) for (x, w) ∈ zip(x_data, w_data)],
    ẋ_data,
    b;
    kwargs...
  )
end

end # module SINDyMethod