@testset "Stag Hunt Game" begin

  sh = GameDynamics.Presets.SH
  x_train_data, ẋ_train_data = sample_with_velocities(
    sh.replicator, 
    [0.125, 0.625],
    collect(0.0:0.1:30.0)
  )
  
  # Solve a SIAR problem and verify that the recovered system is identical to the Replicator Dynamics for a Stag Hunt game.
  @polyvar x[1:2]
  basis = monomials(x, 0:3)
  domain = Domains.product_of_simplices([[x[1]], [x[2]]]) 
  v = sh.marginal_payoffs(x)
  p = separated([1, 1], x_train_data, ẋ_train_data) do i, x_dataᵢ, ẋ_dataᵢ
    SIAR(
      x_dataᵢ, 
      ẋ_dataᵢ, 
      basis;
      SOCP_optimizer=SOCP_optimizer,
      SI_constraints = [
        (model, p) -> SI.simplex_invariance(model, [x[i]], p; domain=domain),
        (model, p) -> SI.positive_correlation(model, p, v[i]; domain=domain, maxdegree=5)
      ],
      η=1e-8
    )
  end
  @test p[1] ≈ x[1] * (1 - x[1]) * (2 * x[2] - 1) atol=ABSOLUTE_TOLERANCE 
  @test p[2] ≈ x[2] * (1 - x[2]) * (2 * x[1] - 1) atol=ABSOLUTE_TOLERANCE

  f̂ = polynomial_system(p, x)
  x₀_test     = [0.375, 0.625]
  t_test_data = collect(0.0:0.1:1.0)
  x_test_data = sample(sh.replicator, x₀_test, t_test_data)
  x̂_test_data = sample(f̂,             x₀_test, t_test_data)
  @test x_test_data ≈ x̂_test_data atol=ABSOLUTE_TOLERANCE

end

@testset "Matching Pennies Game" begin

  mp = GameDynamics.Presets.MP
  x_train_data, ẋ_train_data = sample_with_velocities(
    mp.replicator, 
    [0.125, 0.625], 
    collect(0.0:0.1:30.0)
  )

  # Solve a SIAR problem and verify that the recovered system is qualitatively identical to the Replicator Dynamics for a Matching Pennies game.
  @polyvar x[1:2]
  basis = monomials(x, 0:3)
  domain = Domains.product_of_simplices([[x[1]], [x[2]]]) 
  v = mp.marginal_payoffs(x)
  p = separated([1, 1], x_train_data, ẋ_train_data) do i, x_dataᵢ, ẋ_dataᵢ
    SIAR(
      x_dataᵢ, 
      ẋ_dataᵢ, 
      basis;
      SOCP_optimizer=SOCP_optimizer,
      SI_constraints = [
        (model, p) -> SI.simplex_invariance(model, [x[i]], p; domain=domain),
        (model, p) -> SI.positive_correlation(model, p, v[i]; domain=domain, maxdegree=5)
      ],
      η=1e-8
    )
  end
  @test p[1] ≈ 2 * x[1] * (1 - x[1]) * (2 * x[2] - 1) atol=ABSOLUTE_TOLERANCE
  @test p[2] ≈ 2 * x[2] * (1 - x[2]) * (1 - 2 * x[1]) atol=ABSOLUTE_TOLERANCE

  f̂ = polynomial_system(p, x)
  x₀_test     = [0.375, 0.625]
  t_test_data = collect(0.0:0.1:1.0)
  x_test_data = sample(mp.replicator, x₀_test, t_test_data)
  x̂_test_data = sample(f̂,             x₀_test, t_test_data)
  @test x_test_data ≈ x̂_test_data atol=ABSOLUTE_TOLERANCE

end

@testset "Rock-Paper-Scissors Game" begin

  rps = GameDynamics.Presets.RPS
  x_train_data, ẋ_train_data = sample_with_velocities(
    rps.replicator, 
    [0.125, 0.625, 0.375, 0.25], 
    collect(0.0:0.1:30.0)
  )

  # Solve a SIAR problem and verify that the recovered system is qualitatively identical to the Replicator Dynamics for a Rock-Paper-Scissors game.
  @polyvar x₁[1:2] x₂[1:2]
  x = [x₁, x₂]
  basis = monomials([x₁; x₂], 0:3)
  domain = Domains.product_of_simplices(x) 
  v = rps.marginal_payoffs(x)
  p = separated([2, 2], x_train_data, ẋ_train_data) do i, x_dataᵢ, ẋ_dataᵢ
    SIAR(
      x_dataᵢ, 
      ẋ_dataᵢ, 
      basis;
      SOCP_optimizer=SOCP_optimizer,
      SI_constraints = [
        (model, p) -> SI.simplex_invariance(model, x[i], p; domain=domain),
        (model, p) -> SI.positive_correlation(model, p, v[i]; domain=domain, maxdegree=5)
      ],
      η=1e-8
    )
  end
  u₁ = 
         x₁[1]          * (        - x₂[2] + (1 - x₂[1] - x₂[2])) + 
                 x₁[2]  * (  x₂[1]         - (1 - x₂[1] - x₂[2])) +
    (1 - x₁[1] - x₁[2]) * (- x₂[1] + x₂[2]                      )
  u₂ = 
         x₂[1]          * (        - x₁[2] + (1 - x₁[1] - x₁[2])) + 
                 x₂[2]  * (  x₁[1]         - (1 - x₁[1] - x₁[2])) +
    (1 - x₂[1] - x₂[2]) * (- x₁[1] + x₁[2]                      )
  @test p[1] ≈ x₁[1] * (      - x₂[2] + (1 - x₂[1] - x₂[2]) - u₁) atol=ABSOLUTE_TOLERANCE
  @test p[2] ≈ x₁[2] * (x₂[1]         - (1 - x₂[1] - x₂[2]) - u₁) atol=ABSOLUTE_TOLERANCE
  @test p[3] ≈ x₂[1] * (      - x₁[2] + (1 - x₁[1] - x₁[2]) - u₂) atol=ABSOLUTE_TOLERANCE
  @test p[4] ≈ x₂[2] * (x₁[1]         - (1 - x₁[1] - x₁[2]) - u₂) atol=ABSOLUTE_TOLERANCE

  f̂ = polynomial_system(p, [x₁; x₂])
  x₀_test     = [0.375, 0.125, 0.25, 0.125]
  t_test_data = collect(0.0:0.1:20.0)
  x_test_data = sample(rps.replicator, x₀_test, t_test_data)
  x̂_test_data = sample(f̂,              x₀_test, t_test_data)
  @test x_test_data ≈ x̂_test_data atol=ABSOLUTE_TOLERANCE

end

@testset "Time-Variant Game Dynamics" begin

  @polyvar x₁ x₂ w
  x = [x₁, x₂]
  A₁ = [
     1 + w -1;
    -1      1
  ]
  A₂ = - A₁
  v = marginal_payoffs([A₁, A₂], [[x₁], [x₂]])
  f = replicator_as_polynomial_system([[x₁], [x₂]], v, w)
  x_train_data, w_train_data, ẋ_train_data = sample_with_velocities(
    f, 
    [0.125, 0.625], 
    collect(0.0:0.1:30.0), 
    Dict(
       0.0 => 0.0,
       6.0 => 0.5,
      12.0 => 1.0,
      18.0 => 1.5,
      24.0 => 2.0
    ) # We let the control inputs vary during data collection.
  )

  # Solve a SIAR problem and verify that the recovered system is identical to the original one.
  basis = monomials([x₁, x₂, w], 0:4)
  domain = Domains.product_of_simplices([[x₁], [x₂]]) 
  p = separated(
    [1, 1], 
    x_train_data, 
    w_train_data, 
    ẋ_train_data
  ) do i, x_dataᵢ, w_dataᵢ, ẋ_dataᵢ
    SIAR(
      x_dataᵢ, 
      w_dataᵢ,
      ẋ_dataᵢ, 
      basis;
      SOCP_optimizer=SOCP_optimizer,
      SI_constraints = [
        (model, p) -> SI.simplex_invariance(model, [x[i]], p; domain=domain),
        (model, p) -> SI.positive_correlation(model, p, v[i]; domain=domain, maxdegree=6)
      ],
      η=1e-8
    )
  end
  @test p[1] ≈ x₁ * (1 - x₁) * ((4 + w) * x₂ - 2) atol=ABSOLUTE_TOLERANCE
  @test p[2] ≈ x₂ * (1 - x₂) * (2 - (4 + w) * x₁) atol=ABSOLUTE_TOLERANCE

  f̂ = polynomial_system(p, [x₁, x₂, w])
  x₀_test     = [0.375, 0.625]
  t_test_data = collect(0.0:0.1:1.0)
  w_test_data = Dict(
    0.0 =>  0.0,
    0.2 => -0.1,
    0.4 =>  0.1,
    0.6 =>  0.5,
    0.8 => -1.0
  )
  x_test_data, _ = sample(f, x₀_test, t_test_data, w_test_data)
  x̂_test_data, _ = sample(f̂, x₀_test, t_test_data, w_test_data)
  @test x_test_data ≈ x̂_test_data atol=ABSOLUTE_TOLERANCE

end