using Test
using LinearAlgebra
using DynamicPolynomials
using NormalFormGames
using GameDynamics

ABSOLUTE_TOLERANCE = 1e-4

@testset "GameDynamics.jl" begin

  @polyvar x₁ x₂ w
  x = [[x₁], [x₂]]

  A₁ = [
    -1 + w  1;
     1     -1
  ]
  A₂ = - A₁
  v = marginal_payoffs([A₁, A₂], x)
  f = replicator_as_polynomial_system(x, v, w)

  w_data = Dict(
    0.0 => 0.0,
    0.1 => 0.5,
    0.4 => 1.0,
    0.6 => 1.5,
    0.8 => 2.0
  )
  x_data, w_inferred_data, ẋ_data = sample_with_velocities(
    f, 
    [0.125, 0.625], 
    collect(0.0:0.1:1.0), 
    w_data
  )
  @test x_data[end] ≈ [0.214616, 0.253718] atol=ABSOLUTE_TOLERANCE
  @test w_inferred_data[end] == w_data[0.8]
  @test ẋ_data[end] ≈ [0.251581, -0.297417] atol=ABSOLUTE_TOLERANCE

  # Test the Replicator Dynamics in a Stag Hunt game.
  @testset "Stag Hunt Game" begin

    sh = GameDynamics.Presets.SH

    @polyvar x[1:2]

    p = sh.replicator(x)
    @test p[1] == x[1] * (1 - x[1]) * (2 * x[2] - 1)
    @test p[2] == x[2] * (1 - x[2]) * (2 * x[1] - 1)

    # Verify that the states (0, 0), (0.5, 0.5), and (1, 1) are Nash equilbria.
    @test sh.replicator([0.0, 0.0]) ≈ [0, 0] atol=ABSOLUTE_TOLERANCE
    @test sh.replicator([0.5, 0.5]) ≈ [0, 0] atol=ABSOLUTE_TOLERANCE
    @test sh.replicator([1.0, 1.0]) ≈ [0, 0] atol=ABSOLUTE_TOLERANCE

    # Verify that `V` could be a potential function.
    V(x) = prod(2 * x .- 1) + 1
    x_data = sample(sh.replicator, [0.125, 0.625], collect(0.0:0.1:5.0))
    Vᵢ₋₁, V_data = Iterators.peel(map(V, x_data))
    @test all(
      begin
        d = Vᵢ - Vᵢ₋₁
        Vᵢ₋₁ = Vᵢ
        d > 0
      end
      for Vᵢ ∈ V_data
    )

  end

  # Test the Replicator Dynamics in a Matching Pennies game.
  @testset "Matching Pennies Game" begin

    mp = GameDynamics.Presets.MP

    @polyvar x[1:2]

    p = mp.replicator(x)
    @test p[1] == 2 * x[1] * (1 - x[1]) * (2 * x[2] - 1)
    @test p[2] == 2 * x[2] * (1 - x[2]) * (1 - 2 * x[1])

    # Verify that the uniform distribution is a Nash equilbrium.
    @test mp.replicator([0.5, 0.5]) ≈ [0, 0] atol=ABSOLUTE_TOLERANCE

    # Verify that the KL-Divergence between the uniform distribution and the system's state is time-invariant.
    x_data = sample(mp.replicator, [0.125, 0.625], collect(0.0:0.1:1.0))
    V(x) = - log2(2 ^ 4 * prod(x) * prod(1 .- x)) / 2
    @test all(isapprox(0.642877; atol=ABSOLUTE_TOLERANCE), V.(x_data))

  end

   # Test the Replicator Dynamics in a Rock-Paper-Scissors game.
   @testset "Rock-Paper-Scissors Game" begin

    rps = GameDynamics.Presets.RPS

    @polyvar x₁[1:2] x₂[1:2]

    u₁ = 
           x₁[1]          * (        - x₂[2] + (1 - x₂[1] - x₂[2])) + 
                   x₁[2]  * (  x₂[1]         - (1 - x₂[1] - x₂[2])) +
      (1 - x₁[1] - x₁[2]) * (- x₂[1] + x₂[2]                      )

    u₂ = 
           x₂[1]          * (        - x₁[2] + (1 - x₁[1] - x₁[2])) + 
                   x₂[2]  * (  x₁[1]         - (1 - x₁[1] - x₁[2])) +
      (1 - x₂[1] - x₂[2]) * (- x₁[1] + x₁[2]                      )

    p = rps.replicator([x₁, x₂])
    @test p[1] == x₁[1] * (        - x₂[2] + (1 - x₂[1] - x₂[2]) - u₁)
    @test p[2] == x₁[2] * (          x₂[1] - (1 - x₂[1] - x₂[2]) - u₁)
    @test p[3] == x₂[1] * (- x₁[2]         + (1 - x₁[1] - x₁[2]) - u₂)
    @test p[4] == x₂[2] * (  x₁[1]         - (1 - x₁[1] - x₁[2]) - u₂)

    # Verify that the uniform distribution is a Nash equilbrium.
    @test rps.replicator([1, 1, 1, 1] / 3) ≈ [0, 0, 0, 0] atol=ABSOLUTE_TOLERANCE

    # Verify that the KL-Divergence between the uniform distribution and the system's state is time-invariant.
    x_data = sample(rps.replicator, [0.125, 0.625, 0.375, 0.25], collect(0.0:0.1:1.0))
    V(x) = begin
      x₁ = @view x[1:2]
      x₂ = @view x[3:4]
      - log2(3 ^ 6 * prod(x) * (1 - sum(x₁)) * (1 - sum(x₂))) / 3
    end
    @test all(isapprox(0.332791; atol=ABSOLUTE_TOLERANCE), V.(x_data))

  end

end