module SystemIdentification

using DynamicPolynomials
using LinearAlgebra

export separated, uncoupled, rescale_data!, Domains, SideInformation

# DOCME
const DEFAULT_ZERO_THRESHOLD     = 1e-4

# DOCME
const DEFAULT_ABSOLUTE_TOLERANCE = 1e-6

# DOCME
function separated(
  fn, 
  x_data, 
  w_data, 
  ẋ_data;
  sizes=Iterators.repeated(1, length(first(ẋ_data)))
)
  values = []
  sᵢ₋₁ = 0
  for (i, mᵢ) ∈ pairs(sizes)
    ẋᵢ_data = [(@view ẋₖ[sᵢ₋₁ + 1:sᵢ₋₁ + mᵢ]) for ẋₖ ∈ ẋ_data]
    append!(values, fn(i, x_data, w_data, ẋᵢ_data))
    sᵢ₋₁ += mᵢ
  end
  values
end

function separated(fn, x_data, ẋ_data; kwargs...)
  separated(
    (i, x_data, _, ẋᵢ_data) -> fn(i, x_data, ẋᵢ_data), 
    x_data, 
    nothing, 
    ẋ_data; 
    kwargs...
  )
end

# DOCME
function rescale_data!(D)
  scaling = Float64[]
  sizehint!(scaling, size(D, 2))
  for Dᵢ ∈ eachcol(D)
    scaleingᵢ = norm(Dᵢ)
    Dᵢ ./= scaleingᵢ
    push!(scaling, scaleingᵢ) 
  end
  scaling
end

module Methods

  include("methods/SINDy.jl")
  import .SINDyMethod

  include("methods/SINDySLS.jl")
  import .SINDySLSMethod
  
  include("methods/SIAR.jl")
  import .SIARMethod

  const SINDy    = SINDyMethod.SINDy
  const SINDySLS = SINDySLSMethod.SINDySLS
  const SIAR     = SIARMethod.SIAR

end # module Methods

import .Methods

# DOCME
Methods.SINDyMethod.SINDy

# DOCME
Methods.SINDySLSMethod.SINDySLS

# DOCME
Methods.SIARMethod.SIAR

module Domains

  using DynamicPolynomials
  using SemialgebraicSets
  
  # DOCME
  function simplex(x)
    @assert !isempty(x) "Zero-dimensional simplices are not supported. Please verify that the vectors of variables is non-empty."
    if length(x) == 1
      x₁ = only(x)
      @set x₁ * (1 - x₁) ≥ 0
    else
      BasicSemialgebraicSet(FullSpace(), [x; 1 - sum(x)])
    end
  end

  # DOCME
  function product_of_simplices(x)
    @assert !isempty(x) && !any(map(isempty, x)) "Products of simplices that include one or more zero-dimensional simplices are not supported. Please verify that all the vectors of variables are non-empty."
    function inequalities(xᵢ)
      if length(xᵢ) == 1
        xᵢ₁ = only(xᵢ)
        [xᵢ₁ * (1 - xᵢ₁)]
      else
        [xᵢ; 1 - sum(xᵢ)]
      end
    end
    g = inequalities(x[begin])
    sizehint!(g, sum(mᵢ > 1 ? mᵢ + 1 : 1 for mᵢ ∈ map(length, x)))
    for xᵢ ∈ (@view x[begin + 1:end])
      append!(g, inequalities(xᵢ))
    end
    BasicSemialgebraicSet(FullSpace(), g)
  end

end

module SideInformation

  include("side_information/set_invariance.jl")
  using .SetInvariance

  include("side_information/correlation.jl")
  using .Correlation

end


end # module SystemIdentification
