import requests
import os
from pathlib import Path
from urllib.parse import urlparse
from tqdm import tqdm # Using tqdm for progress bars, similar to dlutils experience

# --- Helper Function to Download a File from URL ---
def download_url(url: str, directory: str):
    """
    Downloads a file from a given URL into the specified directory,
    showing a progress bar.

    Args:
        url: The URL of the file to download.
        directory: The directory path where the file should be saved.
        filename: The name to save the file as.
    """
    try:
        # Ensure the target directory exists
        target_dir = Path(directory)
        target_dir.mkdir(parents=True, exist_ok=True)

        # Extract filename from URL
        parsed_url = urlparse(url)
        filename = os.path.basename(parsed_url.path)
        if not filename:
            # Basic fallback if path is empty, might need refinement based on URLs
            print(f"Warning: Could not automatically determine filename for {url}. Skipping.")
            return False

        destination_path = target_dir / filename

        print(f"Attempting to download: {filename}")
        print(f"  From: {url}")
        print(f"  To:   {destination_path}")

        # Check if file already exists
        if destination_path.exists():
            print(f"File already exists at {destination_path}. Skipping download.")
            return True # Indicate success (file exists)

        # Make the request with streaming enabled
        response = requests.get(url, stream=True, timeout=60) # Increased timeout
        response.raise_for_status()  # Raise HTTPError for bad responses (4xx or 5xx)

        # Get total file size from headers (if available)
        total_size = int(response.headers.get('content-length', 0))

        # Download with progress bar
        block_size = 1024 * 8 # 8 Kibibytes
        progress_bar = tqdm(total=total_size, unit='iB', unit_scale=True, desc=filename)
        with open(destination_path, 'wb') as f:
            for chunk in response.iter_content(chunk_size=block_size):
                progress_bar.update(len(chunk))
                f.write(chunk)
        progress_bar.close()

        # Verify download size
        if total_size != 0 and progress_bar.n != total_size:
            print(f"Error: Downloaded size ({progress_bar.n} bytes) does not match expected size ({total_size} bytes).")
            # Clean up potentially corrupted file
            try:
                os.remove(destination_path)
                print(f"Removed incomplete file: {destination_path}")
            except OSError as e:
                print(f"Error removing incomplete file {destination_path}: {e}")
            return False # Indicate failure

        print(f"Successfully downloaded {filename}.")
        return True # Indicate success

    except requests.exceptions.RequestException as e:
        print(f"\nError downloading {url}: {e}")
        # Clean up partially downloaded file if error occurs
        if 'destination_path' in locals() and destination_path.exists():
             try:
                 os.remove(destination_path)
                 print(f"Removed partially downloaded file: {destination_path}")
             except OSError as oe:
                 print(f"Error removing partial file {destination_path}: {oe}")
        return False # Indicate failure
    except Exception as e:
         print(f"\nAn unexpected error occurred for {url}: {e}")
         if 'destination_path' in locals() and destination_path.exists():
             try:
                 os.remove(destination_path)
                 print(f"Removed partially downloaded file: {destination_path}")
             except OSError as oe:
                 print(f"Error removing partial file {destination_path}: {oe}")
         return False # Indicate failure

# --- Main Download Logic ---

# Note: This implementation directly uses the fallback URLs provided in the
# original code's except blocks. Downloading from Google Drive IDs reliably
# often requires specific handling (cookies, confirmations) best handled by
# libraries like gdown or the original dlutils.

# --- FFHQ ---
print("\n--- Downloading FFHQ artifacts ---")
ffhq_dir = 'training_artifacts/ffhq'
urls_ffhq = [
    'https://alaeweights.s3.us-east-2.amazonaws.com/ffhq/model_submitted.pth',
    'https://alaeweights.s3.us-east-2.amazonaws.com/ffhq/model_194.pth',
    'https://alaeweights.s3.us-east-2.amazonaws.com/ffhq/model_157.pth'
]
for url in urls_ffhq:
    download_url(url, ffhq_dir)

# --- CelebA ---
# print("\n--- Downloading CelebA artifacts ---")
# celeba_dir = 'training_artifacts/celeba'
# url_celeba = 'https://alaeweights.s3.us-east-2.amazonaws.com/celeba/model_final.pth'
# download_url(url_celeba, celeba_dir)

# --- Bedroom ---
# print("\n--- Downloading Bedroom artifacts ---")
# bedroom_dir = 'training_artifacts/bedroom'
# url_bedroom = 'https://alaeweights.s3.us-east-2.amazonaws.com/bedroom/model_final.pth'
# download_url(url_bedroom, bedroom_dir)

# --- CelebA-HQ256 ---
# print("\n--- Downloading CelebA-HQ256 artifacts ---")
# celebahq_dir = 'training_artifacts/celeba-hq256'
# urls_celebahq = [
#     'https://alaeweights.s3.us-east-2.amazonaws.com/celeba-hq256/model_262r.pth',
#     'https://alaeweights.s3.us-east-2.amazonaws.com/celeba-hq256/model_580r.pth'
# ]
# for url in urls_celebahq:
#     download_url(url, celebahq_dir)

print("\nDownload process completed.")