import numpy as np

class LinearModel:
    def __init__(self,
                 lag=1,
                 horizon=1):
        
        self.lag = lag
        self.horizon = horizon
        self.parameter_matrix = np.random.random((horizon,lag))
        self.x_ = 0
        self.u_mean = None # TD
        self.iter = 0 # keeps track of the number of iterations internally to update the average TD

        ## for online training with RLS
        self.precision_matrix = np.eye(lag)

    def forecast(self,x,lagged_values):
        # update TD
        self.iter += 1
        self._update_average_td(x,self.iter)
        # prediction
        predicted_values = self.parameter_matrix @ lagged_values
        predicted_time_stamps = np.array( [i*self.u_mean for i in range(1,self.horizon+1)] ) + x
        return predicted_time_stamps, predicted_values

    def train_model(self,training_values):
        L,R = self._construct_training_matrices(training_values)
        A = np.linalg.pinv(L.T) @ R.T
        self.parameter_matrix = A.T

    def _update_average_td(self,x,t):
        u = x-self.x_
        if self.u_mean is None:
            self.u_mean = x - self.x_
        else:
            self.u_mean = (t-1)/t * self.u_mean + 1/t * u
        self.x_ = x

    def update_parameters(self,lagged_values,reference_values):
        lagged_values = np.expand_dims(lagged_values,axis=1)
        reference_values = np.expand_dims(reference_values,axis=1)
        self.precision_matrix = self.precision_matrix - self.precision_matrix @ lagged_values @ lagged_values.T @ self.precision_matrix / ( 1 + lagged_values.T @ self.precision_matrix @ lagged_values )
        A = self.parameter_matrix.T + self.precision_matrix @ lagged_values @ (reference_values.T - lagged_values.T @ self.parameter_matrix.T)
        self.parameter_matrix = A.T

    def _construct_training_matrices(self,signal_values):
        p = self.lag
        h = self.horizon
        
        T = len(signal_values)
        R = np.zeros((h,T-h-p+1))
        L = np.zeros((p,T-h-p+1))

        for t in range(p,T-h+1):
            i = t-p # column index
            L[:,i] = signal_values[t-p:t] 
            R[:,i] = signal_values[t:t+h]

        return L,R

class ZeroOrderHoldModel:
    def __init__(self,
                horizon=1):
        
        self.horizon = horizon
        self.x_ = 0
        self.u_mean = None
        self.iter = 0

    def _update_average_td(self,x,t):
        u = x-self.x_
        if self.u_mean is None:
            self.u_mean = x - self.x_
        else:
            self.u_mean = (t-1)/t * self.u_mean + 1/t * u
        self.x_ = x

    def forecast(self,x,y):
        # update TD
        self.iter += 1
        self._update_average_td(x,self.iter)
        # prediction
        hold_values = np.array( [y]*self.horizon )
        predicted_time_stamps = np.array( [i*self.u_mean for i in range(1,self.horizon+1)] ) + x
        return predicted_time_stamps, hold_values
