# docs and experiment results can be found at https://docs.cleanrl.dev/rl-algorithms/ppo/#ppo_continuous_actionpy
import argparse
import os
import random
import time
import datetime
from distutils.util import strtobool

import numpy as np
import torch
import torch.nn as nn
import torch.optim as optim
import d4rl
from torch.utils.tensorboard import SummaryWriter

from utils import SuccessEvaluator, RunningMeanStd, RewardForwardFilter, select_gpu, make_env
from agents import Agent


def parse_args():
    # fmt: off
    parser = argparse.ArgumentParser()
    parser.add_argument("--exp-name", type=str, default=os.path.basename(__file__).rstrip(".py"),
        help="the name of this experiment")
    parser.add_argument("--tensorboard-log", type=str, default="runs_ppo",
        help="the log location for tensorboard (if None, no logging)")
    parser.add_argument("--device", type=str, default=None)
    parser.add_argument("--seed", type=int, default=1,
        help="seed of the experiment")
    parser.add_argument("--torch-deterministic", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="if toggled, `torch.backends.cudnn.deterministic=False`")
    parser.add_argument("--cuda", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="if toggled, cuda will be enabled by default")
    parser.add_argument("--track", type=lambda x: bool(strtobool(x)), default=False, nargs="?", const=True,
        help="if toggled, this experiment will be tracked with Weights and Biases")
    parser.add_argument("--wandb-project-name", type=str, default="cleanRL",
        help="the wandb's project name")
    parser.add_argument("--wandb-entity", type=str, default=None,
        help="the entity (team) of wandb's project")
    parser.add_argument("--capture-video", type=lambda x: bool(strtobool(x)), default=False, nargs="?", const=True,
        help="whether to capture videos of the agent performances (check out `videos` folder)")
    parser.add_argument("--success-num", type=int, default=10,
        help="the number of steps required for success")

    # Algorithm specific arguments
    parser.add_argument("--algo", type=str, default="drnd",
        help="the algorithm to use: ppo | rnd | drnd | cfn")
    parser.add_argument("--env-id", type=str, default="door-v0",
        help="the id of the environment")
    parser.add_argument("--total-timesteps", type=int, default=1000000,
        help="total timesteps of the experiments")
    parser.add_argument("--learning-rate", type=float, default=3e-4,
        help="the learning rate of the optimizer")
    parser.add_argument("--num-envs", type=int, default=1,
        help="the number of parallel game environments")
    parser.add_argument("--num-steps", type=int, default=500,
        help="the number of steps to run in each environment per policy rollout")
    parser.add_argument("--anneal-lr", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="Toggle learning rate annealing for policy and value networks")
    parser.add_argument("--gamma", type=float, default=0.99,
        help="the discount factor gamma")
    parser.add_argument("--gae-lambda", type=float, default=0.95,
        help="the lambda for the general advantage estimation")
    parser.add_argument("--num-minibatches", type=int, default=32,
        help="the number of mini-batches")
    parser.add_argument("--update-epochs", type=int, default=10,
        help="the K epochs to update the policy")
    parser.add_argument("--norm-adv", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="Toggles advantages normalization")
    parser.add_argument("--clip-coef", type=float, default=0.2,
        help="the surrogate clipping coefficient")
    parser.add_argument("--clip-vloss", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="Toggles whether or not to use a clipped loss for the value function, as per the paper.")
    parser.add_argument("--ent-coef", type=float, default=0.0,
        help="coefficient of the entropy")
    parser.add_argument("--vf-coef", type=float, default=0.5,
        help="coefficient of the value function")
    parser.add_argument("--max-grad-norm", type=float, default=0.5,
        help="the maximum norm for the gradient clipping")
    parser.add_argument("--target-kl", type=float, default=None,
        help="the target KL divergence threshold")
    
    # RND specific arguments
    parser.add_argument("--update-proportion", type=float, default=0.25,
        help="the proportion of the batch to use for RND update")
    parser.add_argument("--num-iterations-obs-norm-init", type=int, default=10,
        help="number of iterations to initialize the observations normalization parameters")
    parser.add_argument("--ext-coef", type=float, default=1.0,
        help="coefficient of the extrinsic reward")
    parser.add_argument("--int-coef", type=float, default=1.0,
        help="coefficient of the intrinsic reward")
    parser.add_argument("--alpha", type=float, default=0.99,
        help="DRND alpha")

    # Environment specific arguments
    parser.add_argument("--fixed-start", type=bool, default=True,
        help="whether to use fixed start position")
    parser.add_argument("--distance-threshold", type=float, default=0.05)
    parser.add_argument("--mode", default=0)
    args = parser.parse_args()
    args.batch_size = int(args.num_envs * args.num_steps)
    args.minibatch_size = int(args.batch_size // args.num_minibatches)
    # fmt: on
    return args


if __name__ == "__main__":
    args = parse_args()
    if "Fetch" in args.env_id:
        import gymnasium as gym
    else:
        import gym
    evaluator = SuccessEvaluator(10)
    run_name = f"{args.env_id}__{args.exp_name}__{args.seed}__{datetime.datetime.now().strftime('%d_%m_%Y_%H_%M_%S')}"
    if args.track:
        import wandb
        wandb.init(
            project=args.wandb_project_name,
            entity=args.wandb_entity,
            sync_tensorboard=True,
            config=vars(args),
            name=run_name,
            monitor_gym=True,
            save_code=True,
        )
    writer = SummaryWriter(f"{args.tensorboard_log}/{datetime.datetime.now().strftime('%Y_%m_%d')}/{args.env_id}/{run_name}")
    writer.add_text(
        "hyperparameters",
        "|param|value|\n|-|-|\n%s" % ("\n".join([f"|{key}|{value}|" for key, value in vars(args).items()])),
    )

    # TRY NOT TO MODIFY: seeding
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = args.torch_deterministic

    gpu_id = select_gpu() if args.device is None else args.device
    device = torch.device(f"cuda:{gpu_id}" if torch.cuda.is_available() and args.cuda else "cpu")

    # env setup
    envs = gym.vector.SyncVectorEnv(
        [make_env(args.env_id, i, args.capture_video, run_name, args.gamma, args.mode, args.distance_threshold) for i in range(args.num_envs)]
    )
    assert isinstance(envs.single_action_space, gym.spaces.Box), "only continuous action space is supported"

    agent = Agent[args.algo](envs, args).to(device)
    optimizer = optim.Adam(agent.parameters(), lr=args.learning_rate, eps=1e-5)


    # normalize observation
    if args.algo in ["RND", "rnd", "DRND", "drnd", "CFN", "cfn"]:
        print('Initializes observation normalization...')
        if "Fetch" in args.env_id:
            next_obs, _ = envs.reset(seed=args.seed)
        else:
            envs.seed(args.seed)
            next_obs = envs.reset()
        next_obs = torch.Tensor(next_obs).to(device)
        next_done = torch.zeros(args.num_envs).to(device)
        num_updates = args.total_timesteps // args.batch_size
        next_obss = []
        obs_rms = RunningMeanStd(shape=envs.single_observation_space.shape)
        for step in range(args.num_steps * args.num_iterations_obs_norm_init):
            actions = np.random.randint(envs.action_space.low, envs.action_space.high, size=(args.num_envs,))
            if "Fetch" in args.env_id:
                next_obs, reward, terminated, truncated, infos = envs.step(actions)
                done = np.logical_or(terminated, truncated)
            else:
                next_obs, reward, done, infos = envs.step(actions)
            if len(next_obss) % (args.num_steps * args.num_envs) == 0:
                next_obss = np.stack(next_obs)
                next_obss = next_obss.reshape((-1,) + envs.single_observation_space.shape)
                obs_rms.update(next_obss)
                next_obss = []
        print('Observation normalization initialized.')

    # ALGO Logic: Storage setup
    obs = torch.zeros((args.num_steps, args.num_envs) + envs.single_observation_space.shape).to(device)
    actions = torch.zeros((args.num_steps, args.num_envs) + envs.single_action_space.shape).to(device)
    logprobs = torch.zeros((args.num_steps, args.num_envs)).to(device)
    rewards = torch.zeros((args.num_steps, args.num_envs)).to(device)
    if args.algo in ["RND", "rnd", "DRND", "drnd", "CFN", "cfn"]:
        rndloss = torch.zeros((args.num_steps, args.num_envs)).to(device)
    dones = torch.zeros((args.num_steps, args.num_envs)).to(device)
    values = torch.zeros((args.num_steps, args.num_envs)).to(device)

    # TRY NOT TO MODIFY: start the game
    global_step = 0
    start_time = time.time()
    if "Fetch" in args.env_id:
        next_obs, _ = envs.reset(seed=args.seed)
    else:
        envs.seed(args.seed)
        next_obs = envs.reset()
    next_obs = torch.Tensor(next_obs).to(device)
    next_done = torch.zeros(args.num_envs).to(device)
    num_updates = args.total_timesteps // args.batch_size
    reward_rms = RunningMeanStd()
    discounted_reward = RewardForwardFilter(args.gamma)
    for update in range(1, num_updates + 1):
        # Annealing the rate if instructed to do so.
        if args.anneal_lr:
            frac = 1.0 - (update - 1.0) / num_updates
            lrnow = frac * args.learning_rate
            optimizer.param_groups[0]["lr"] = lrnow
        episode_steps = 0
        success_count = 0
        for step in range(0, args.num_steps):
            global_step += 1 * args.num_envs
            obs[step] = next_obs
            dones[step] = next_done

            # ALGO LOGIC: action logic
            with torch.no_grad():
                action, logprob, _, value = agent.get_action_and_value(next_obs)
                values[step] = value.flatten()
            actions[step] = action
            logprobs[step] = logprob

            # TRY NOT TO MODIFY: execute the game and log data.
            if "Fetch" in args.env_id:
                next_obs, reward, terminated, truncated, infos = envs.step(action.cpu().numpy())
                done = np.logical_or(terminated, truncated)
            else:
                next_obs, reward, done, infos = envs.step(action.cpu().numpy())
            rewards[step] = torch.tensor(reward).to(device).view(-1)
            next_obs, next_done = torch.Tensor(next_obs).to(device), torch.Tensor(done).to(device)
            if args.algo in ["RND", "rnd"]:
                rndloss[step] = agent.get_rnd_loss(next_obs)
            elif args.algo in ["DRND", "drnd", "CFN", "cfn"]:
                rndloss[step] = agent.get_rnd_reward(next_obs).view(-1)
            if done:
                episode_steps = step + 1
            # Only print when at least 1 env is done
            if "Fetch" not in args.env_id:
                if done:
                    # Skip the envs that are not done
                    episode_steps = step+1
                    success = 0
                    if infos is None:
                        continue
                    if success_count > args.success_num:
                        success=1
                        success_count = 0
                    return_per_episode = np.mean([info['episode']['r'] for info in infos if info is not None])
                    length_per_episode = np.mean([info['episode']['l'] for info in infos if info is not None])
                    writer.add_scalar("charts/episodic_return", return_per_episode, global_step)
                    writer.add_scalar("charts/episodic_length", length_per_episode, global_step)
                    writer.add_scalar("charts/success_rate",success,global_step)
                    writer.add_scalar("charts/success_rate_final",float(infos[0]['goal_achieved']),global_step)
                    envs.seed(args.seed)
                    next_obs = envs.reset()
                    next_obs = torch.Tensor(next_obs).to(device)
            else:
                if "final_info" not in infos:
                    continue
                for info in infos["final_info"]:
                    # Skip the envs that are not done
                    if info is None:
                        continue
                    success_rate = evaluator.eval_success(info)
                    if success_rate is not None:
                        print(f"global_step={global_step}, \033[92msuccess_rate={success_rate}\033[0m")
                        writer.add_scalar("charts/success_rate", success_rate, global_step)
                        writer.add_scalar("charts/episodic_return", info["episode"]["r"], global_step)
                        writer.add_scalar("charts/episodic_length", info["episode"]["l"], global_step)
                if args.fixed_start:
                    if "Fetch" in args.env_id:
                        next_obs, _ = envs.reset(seed=args.seed)
                    else:
                        envs.seed(args.seed)
                        next_obs = envs.reset()
                    next_obs = torch.Tensor(next_obs).to(device)
        if args.algo in ["RND", "rnd", "DRND", "drnd", "CFN", "cfn"]:
            # compute rnd return
            rnd_return = np.array(
                [discounted_reward.update(reward_per_step) for reward_per_step in rndloss.cpu().data.numpy()[: episode_steps][::-1]]
            )[::-1]
            # update rnd return rms
            reward_rms.update_from_moments(
                    rnd_return.mean(), 
                    rnd_return.var(), 
                    episode_steps
                )
            # update rnd loss
            rndloss /= np.sqrt(reward_rms.var)
            # update reward
            rewards = args.ext_coef * rewards + args.int_coef * rndloss

            # update obs rms
            obs_rms.update(obs.cpu().numpy().reshape((-1,) + envs.single_observation_space.shape))

        # bootstrap value if not done
        with torch.no_grad():
            next_value = agent.get_value(next_obs).reshape(1, -1)
            advantages = torch.zeros_like(rewards).to(device)
            lastgaelam = 0
            for t in reversed(range(args.num_steps)):
                if t == args.num_steps - 1:
                    nextnonterminal = 1.0 - next_done
                    nextvalues = next_value
                else:
                    nextnonterminal = 1.0 - dones[t + 1]
                    nextvalues = values[t + 1]
                delta = rewards[t] + args.gamma * nextvalues * nextnonterminal - values[t]
                advantages[t] = lastgaelam = delta + args.gamma * args.gae_lambda * nextnonterminal * lastgaelam
            returns = advantages + values

        # flatten the batch
        b_obs = obs.reshape((-1,) + envs.single_observation_space.shape)
        b_logprobs = logprobs.reshape(-1)
        b_actions = actions.reshape((-1,) + envs.single_action_space.shape)
        b_advantages = advantages.reshape(-1)
        b_returns = returns.reshape(-1)
        b_values = values.reshape(-1)

        # Optimizing the policy and value network
        b_inds = np.arange(args.batch_size)
        clipfracs = []
        for epoch in range(args.update_epochs):
            np.random.shuffle(b_inds)
            for start in range(0, args.batch_size, args.minibatch_size):
                end = start + args.minibatch_size
                mb_inds = b_inds[start:end]

                _, newlogprob, entropy, newvalue = agent.get_action_and_value(b_obs[mb_inds], b_actions[mb_inds])
                logratio = newlogprob - b_logprobs[mb_inds]
                ratio = logratio.exp()

                with torch.no_grad():
                    # calculate approx_kl http://joschu.net/blog/kl-approx.html
                    old_approx_kl = (-logratio).mean()
                    approx_kl = ((ratio - 1) - logratio).mean()
                    clipfracs += [((ratio - 1.0).abs() > args.clip_coef).float().mean().item()]

                mb_advantages = b_advantages[mb_inds]
                if args.norm_adv:
                    mb_advantages = (mb_advantages - mb_advantages.mean()) / (mb_advantages.std() + 1e-8)

                # Policy loss
                pg_loss1 = -mb_advantages * ratio
                pg_loss2 = -mb_advantages * torch.clamp(ratio, 1 - args.clip_coef, 1 + args.clip_coef)
                pg_loss = torch.max(pg_loss1, pg_loss2).mean()

                # Value loss
                newvalue = newvalue.view(-1)
                if args.clip_vloss:
                    v_loss_unclipped = (newvalue - b_returns[mb_inds]) ** 2
                    v_clipped = b_values[mb_inds] + torch.clamp(
                        newvalue - b_values[mb_inds],
                        -args.clip_coef,
                        args.clip_coef,
                    )
                    v_loss_clipped = (v_clipped - b_returns[mb_inds]) ** 2
                    v_loss_max = torch.max(v_loss_unclipped, v_loss_clipped)
                    v_loss = 0.5 * v_loss_max.mean()
                else:
                    v_loss = 0.5 * ((newvalue - b_returns[mb_inds]) ** 2).mean()

                entropy_loss = entropy.mean()
                loss = pg_loss - args.ent_coef * entropy_loss + v_loss * args.vf_coef

                if args.algo in ["RND", "rnd", "DRND", "drnd", "CFN", "cfn"]:
                    forward_loss = agent.get_rnd_loss(b_obs[mb_inds])
                    mask = torch.rand(len(forward_loss), device=device)
                    mask = (mask < args.update_proportion).type(torch.FloatTensor).to(device)
                    forward_loss = (forward_loss * mask).sum() / torch.max(
                        mask.sum(), torch.tensor([1], device=device, dtype=torch.float32)
                    )
                    loss += forward_loss[0]
                optimizer.zero_grad()
                loss.backward()
                nn.utils.clip_grad_norm_(agent.parameters(), args.max_grad_norm)
                optimizer.step()

            if args.target_kl is not None:
                if approx_kl > args.target_kl:
                    break

        y_pred, y_true = b_values.cpu().numpy(), b_returns.cpu().numpy()
        var_y = np.var(y_true)
        explained_var = np.nan if var_y == 0 else 1 - np.var(y_true - y_pred) / var_y

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        writer.add_scalar("charts/learning_rate", optimizer.param_groups[0]["lr"], global_step)
        writer.add_scalar("losses/value_loss", v_loss.item(), global_step)
        writer.add_scalar("losses/policy_loss", pg_loss.item(), global_step)
        writer.add_scalar("losses/entropy", entropy_loss.item(), global_step)
        writer.add_scalar("losses/old_approx_kl", old_approx_kl.item(), global_step)
        writer.add_scalar("losses/approx_kl", approx_kl.item(), global_step)
        writer.add_scalar("losses/clipfrac", np.mean(clipfracs), global_step)
        writer.add_scalar("losses/explained_variance", explained_var, global_step)
        writer.add_scalar("charts/SPS", int(global_step / (time.time() - start_time)), global_step)

    envs.close()
    writer.close()