import json
import random
import re
from datetime import datetime
from pathlib import Path
from typing import List, Optional

from zendo.game.rule_evaluator import evaluate
from zendo.game.rule_grammar import get_ast_tree, parser
from zendo.utils import get_env

structure_pattern = r"^[aAbB\\.]{6}$"


class Zendo:
    def __init__(self, rule: str) -> None:
        """
        Instantiate a Zendo game.

        :param rule: the rule to use in this game. It must follow the grammar specified
                     in rule_grammar.py
        """
        self.rule = rule
        self.ast = parser.parse(rule)
        self.expression_regex = re.compile(structure_pattern)

    def evaluate(self, structure: str) -> bool:
        """
        Evaluate the structure against the rule in this game

        :param structure: the structure to evaluate that must respect the structure_pattern

        :return: True if the structure respects the rule, False otherwise
        """
        structure = "".join(structure.split())
        if self.expression_regex.match(structure) is None:
            raise ValueError("Structure is not valid!")
        return evaluate(self.ast, structure)

    def __repr__(self) -> str:
        return get_ast_tree(self.ast)


class Event:
    def __init__(self, event_name: str, value: str, result: bool):
        assert event_name in ["guess", "structure"]

        self.event_name = event_name
        self.value = value
        self.result = result


class ZendoGameSession:
    def __init__(self, dataset_name: str, rule: str, log_file: Optional[Path] = None):
        self.dataset_name = dataset_name
        self.log_file = log_file
        self.zendo = Zendo(rule=rule)
        self.rule = rule
        self.events = []

        self.initial_structures = []
        self._initial_structures()

    def _initial_structures(self) -> None:
        structures = (
            (Path(get_env(self.dataset_name)) / "structures.txt")
            .read_text()
            .split("\n")
        )
        random.shuffle(structures)
        white = False
        black = False
        for x in structures:
            x_label = self.zendo.evaluate(x)
            if x_label and not white:
                self.initial_structures.append((x, x_label))
                white = True
            elif not x_label and not black:
                self.initial_structures.append((x, x_label))
                black = True
            elif white and black:
                break

    def guess_rule(self, guessed_rule: str) -> bool:
        if guessed_rule in [x.value for x in self.events if x.event_name == "guess"]:
            raise FileExistsError("Structure already on the table")

        # Validate rule
        parser.parse(guessed_rule)
        guessed_rule_label = guessed_rule.strip() == self.rule.strip()
        self.events.append(
            Event(event_name="guess", value=guessed_rule, result=guessed_rule_label,)
        )
        return guessed_rule_label

    def build_structure(self, structure: str) -> bool:
        if structure in [x.value for x in self.events if x.event_name == "structure"]:
            raise FileExistsError("Structure already on the table")

        structure_label = self.zendo.evaluate(structure)
        self.events.append(
            Event(event_name="structure", value=structure, result=structure_label,)
        )
        return structure_label

    @staticmethod
    def _format_structure(structure: str, label: bool) -> str:
        structure = " ".join((f"`{x}`" for x in structure))
        label = "◯" if label else "⬤"
        return f"**{structure}** ⟶ **{label}**"

    def get_table(self) -> List[str]:
        return [
            self._format_structure(event.value, event.result)
            for event in self.events
            if event.event_name == "structure"
        ]

    def get_initial_structures(self) -> List[str]:
        return [
            self._format_structure(s, label) for s, label in self.initial_structures
        ]

    def restart(self, rule: str) -> None:
        self.zendo = Zendo(rule=rule)
        self.events.clear()

    def save(self) -> str:
        game_session_info = {
            "timestamp": datetime.today().strftime("%Y-%m-%d-%H:%M:%S"),
            "rule": self.rule,
            "initial_structures": self.initial_structures,
            "events": [x.__dict__ for x in self.events],
        }
        game_log = json.dumps(game_session_info)

        if self.log_file:
            with self.log_file.open(mode="a") as f:
                game_log_line = f"{json.dumps(game_session_info)}\n"
                f.write(game_log_line)

        return game_log
