import random
from pathlib import Path

import streamlit as st
from zendo.game.game import ZendoGameSession
from zendo.game.rule_grammar import zendo_grammar
from zendo.ui.session_state import save_or_get
from zendo.utils import get_env, load_envs

st.title("Binary Zendo")

load_envs()
dataset_name = "DATASET_S6_startPROP"
root = Path(get_env(dataset_name))


s = (root / "rules.txt").read_text().split("\n")
session = save_or_get(
    game=ZendoGameSession(
        dataset_name=dataset_name,
        rule=random.choice(s),
        log_file=Path(get_env("GAME_LOG_FILE")),
    ),
    can_win=True,
)

st.markdown("""A secret rule has been chosen! Will you be able to guess it?""")


st.header("Initial structures")
for x in reversed(session.game.get_initial_structures()):
    st.markdown(x)

if len(session.game.get_initial_structures()) <= 1:
    st.warning(
        "This rule is degenerate! "
        "It's not possible to provide one white and one black example!"
        "(refresh the page for a new rule)"
    )

st.header("Build a structure!")

st.markdown(
    """
Structure grammar: `^[aAbB\\.]{6}$`

Structure recap:
- `red` => `{A, B}`
- `blue` => `{a, b}` 
- `pyramid` => `{a, A}`
- `block` => `{b, B}`

Structure example: `a` `A` `.` `.` `B` `b`"""
)


structure = "".join(st.text_input("Build a structure:").strip().split())
if st.button("Submit structure"):
    try:
        a = session.game.build_structure(structure)
    except ValueError:
        st.info("Structure not valid!")
    except FileExistsError:
        st.info("Structure already on the table!")


st.header("Table")
for x in reversed(session.game.get_table()):
    st.markdown(x)

if not session.game.get_table():
    st.info("The table is empty!")

st.header("Guess a rule!")

st.markdown(
    f"""
Rule grammar:
```
{zendo_grammar}
```

Rule example: `at_least 1 blue block`
"""
)

guessed_rule = " ".join(st.text_input("Guess `exactly` a rule!").strip().split())
if st.button("Submit rule guess"):
    try:
        guessed_rule_label = session.game.guess_rule(guessed_rule)
        if guessed_rule_label:
            if session.can_win:
                st.success("You won!")
                st.balloons()
                session.game.save()
                st.info("Game logged!")
            else:
                st.error("You won, cheater!")
            st.info("A new game started!")
            session.game = ZendoGameSession(
                dataset_name=dataset_name,
                rule=random.choice(s),
                log_file=Path(get_env("GAME_LOG_FILE")),
            )
            session.can_win = True
        else:
            st.error("That's not it. Try again!")
    except ValueError:
        st.info("Rule not valid!")
    except FileExistsError:
        st.error("Rule already tried, it is wrong!")


st.markdown("---")

if st.button("Cheat, lose your honor, and look at the rule"):
    st.write(session.game.zendo.rule)
    session.can_win = False
