# -> BUILD CLI

from getpass import getpass

import fire

from zendo.game.game import Zendo


class ZendoApp:
    """
    Rule grammar:
        RULE     -> PROP 'and' PROP | PROP 'or' PROP | PROP
        PROP     -> QUANTITY OBJ | QUANTITY OBJ REL OBJ
        REL      -> 'touching' | 'surrounded_by' | 'at_the_right_of' | 'at_the_left_of'
        QUANTITY -> 'zero' | 'exactly' NUM | 'at_least' NUM | 'at_most' NUM
        NUM      -> '1' | '2'
        OBJ      -> SHAPE | COLOR | COLOR SHAPE
        COLOR    -> 'red' | 'blue'
        SHAPE    -> 'pyramid' | 'block'

    Expression grammar:
        ^[aAbB\\.]{6}$

    Expression recap:
        red => uppercase
        blu => lowercase
        pyramid => {a, A}
        block => {b, B}
    """

    def play(self, verbose: bool = True) -> None:
        """ Play binary Zendo!

        Insert a rule and try some structures to see if they match the rule!
        An empty structure ends the current rule and start a new one.

        :param verbose: if True print the rule and its AST
        """
        print(self.__doc__)
        while True:
            try:
                s = input("(rule) > ") if verbose else getpass("(rule) > ")
            except EOFError:
                break

            try:
                zendo = Zendo(s)
                if verbose:
                    print(f"\n(ast) >\n{zendo}\n")

                while True:
                    try:
                        e = input("(structure) > ")
                    except EOFError:
                        break

                    try:
                        result = zendo.evaluate(e)
                    except ValueError:
                        print("Structure is not valid, insert a new structure!\n")
                        continue

                    print("White!\n" if result else "Black!\n")

            except ValueError:
                print("Rule is not valid, insert a new rule!\n")


if __name__ == "__main__":
    fire.Fire(ZendoApp)