import os
import torch
import torch.nn.functional as F
from torch.optim import Adam
from core.utils import soft_update, hard_update
import numpy as np


class MultiSAC(object):
    def __init__(self, args, model_constructor):

        self.gamma = args.gamma
        self.tau = args.tau
        self.alpha = args.alpha
        self.writer = args.writer
        self.num_agents = args.num_agents
        self.args = args
        self.model_constructor = model_constructor

        self.target_update_interval = args.target_update_interval
        self.automatic_entropy_tuning = False

        self.device = torch.device("cuda" if torch.cuda.is_available() else "cpu")

        self.policies = [model_constructor.make_model('Gaussian_FF').to(device=self.device) for _ in range(args.num_agents)]
        self.policy_optim = [Adam(self.policies[agent_id].parameters(), lr=args.actor_lr) for agent_id in range(args.num_agents)]

        self.critic = [model_constructor.make_model('Tri_Head_Q').to(device=self.device) for _ in range(args.num_agents)]
        self.critic_optim = [Adam(self.critic[agent_id].parameters(), lr=args.critic_lr) for agent_id in range(args.num_agents)]

        self.critic_target = [model_constructor.make_model('Tri_Head_Q').to(device=self.device) for _ in range(args.num_agents)]
        for c, ct in zip(self.critic, self.critic_target):
            hard_update(ct, c)

        self.auto_reward_critics = []
        for _ in range(args.evo_popn_size):
            self.auto_reward_critics.append([model_constructor.make_model('Tri_Head_Q').to(device=self.device) for _ in range(args.num_agents)])

        self.auto_reward_target_critics = []
        for _ in range(args.evo_popn_size):
            self.auto_reward_target_critics.append([model_constructor.make_model('Tri_Head_Q').to(device=self.device)
             for _ in range(args.num_agents)])

        # Target Entropy = −dim(A) (e.g. , -6 for HalfCheetah-v2) as given in the paper
        if self.automatic_entropy_tuning:
            self.target_entropy = -torch.prod(torch.Tensor(1, args.action_dim)).cuda().item()
            self.log_alpha = torch.zeros(1, requires_grad=True)
            self.alpha_optim = Adam([self.log_alpha], lr=args.alpha_lr)
            self.log_alpha.cuda()



        self.num_updates = 0

        # Statistics Tracker
        self.entropy = {'min': [], 'max': [], 'mean': [], 'std': []}
        self.next_entropy = {'min': [], 'max': [], 'mean': [], 'std': []}
        self.policy_q = {'min': [], 'max': [], 'mean': [], 'std': []}
        self.critic_loss = {'min': [], 'max': [], 'mean': [], 'std': []}

    def compute_stats(self, tensor, tracker):
        """Computes stats from intermediate tensors

             Parameters:
                   tensor (tensor): tensor
                   tracker (object): logger

             Returns:
                   None


         """
        tracker['min'] = torch.min(tensor).item()
        tracker['max'] = torch.max(tensor).item()
        tracker['mean'] = torch.mean(tensor).item()
        tracker['std'] = torch.std(tensor).item()

    def autoreward_update(self, buffer, reward_recipe, policy, agent_id, popn_id, args, return_policies):
        policy.to(device=self.device)
        # critic = self.model_constructor.make_model('Tri_Head_Q').to(device=self.device)
        # critic_target = self.model_constructor.make_model('Tri_Head_Q').to(device=self.device)
        policy_optim = Adam(policy.parameters(), args.actor_lr)
        critic_optim = Adam(self.auto_reward_critics[popn_id][agent_id].parameters(), args.critic_lr)
        # Start Learning
        for _ in range(args.autoreward_iterations):
            state_batch, action_batch, next_state_batch , reward_batch, done_batch = buffer.sample(agent_id)
            #Compute reward
            reward_batch = reward_recipe.compute_reward(torch.cat([state_batch, action_batch, next_state_batch], 1), popn_id).to(device=self.device)
            state_batch = state_batch.cuda()
            next_state_batch = next_state_batch.cuda()
            action_batch = action_batch.cuda()
            done_batch  = done_batch.cuda()
            with torch.no_grad():
                next_state_action, next_state_log_pi,_,_,_= policy.noisy_action(next_state_batch, return_only_action=False)
                qf1_next_target, qf2_next_target, _ =  self.auto_reward_target_critics[popn_id][agent_id](next_state_batch, next_state_action)
                min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - self.alpha * next_state_log_pi
                next_q = reward_batch + self.gamma * (min_qf_next_target) * (1 - done_batch)
            qf_1, qf_2, _ = self.auto_reward_critics[popn_id][agent_id].forward(state_batch, action_batch)  # Two Q-functions to mitigate positive bias in the policy improvement step

            qf1_loss = F.mse_loss(qf_1, next_q)
            qf2_loss = F.mse_loss(qf_2, next_q)
            total_loss = qf1_loss + qf2_loss
            critic_optim.zero_grad()
            total_loss.backward()
            critic_optim.step()

            ############### Policy Update ###########
            pi, log_pi, _,_,_ = policy.noisy_action(state_batch, return_only_action=False)
            qf1_pi, qf2_pi, _ = self.auto_reward_critics[popn_id][agent_id].forward(state_batch, pi)
            min_qf_pi = torch.min(qf1_pi, qf2_pi)
            policy_loss = ((self.alpha * log_pi) - min_qf_pi).mean()
            policy_optim.zero_grad()
            policy_loss.backward()
            policy_optim.step()
            soft_update(self.auto_reward_target_critics[popn_id][agent_id], self.auto_reward_critics[popn_id][agent_id], self.tau)

        return_policies[popn_id][agent_id] = policy.cpu()

    def reset_critic(self, popn_id, agent_id):
        new_net = self.model_constructor.make_model('Tri_Head_Q').to(device=self.device)
        hard_update(self.auto_reward_critics[popn_id][agent_id], new_net)
        hard_update(self.auto_reward_target_critics[popn_id][agent_id], new_net)

    def preserve_critic(self, agent_id, new_elite, old_elite):
        hard_update(self.auto_reward_target_critics[new_elite][agent_id], self.auto_reward_target_critics[old_elite][agent_id])
        hard_update(self.auto_reward_critics[new_elite][agent_id], self.auto_reward_critics[old_elite][agent_id])


    def global_update(self, reward_scaling, buffer):
        for agent_id in range(self.num_agents):
            state_batch, action_batch, next_state_batch , reward_batch, done_batch = buffer.sample(agent_id)
            state_batch = state_batch.cuda()
            action_batch = action_batch.cuda()
            next_state_batch = next_state_batch.cuda()
            reward_batch  = reward_batch.cuda()
            done_batch  = done_batch.cuda()
            with torch.no_grad():
                next_state_action, next_state_log_pi, _, _, _ = self.policies[agent_id].noisy_action(next_state_batch, return_only_action=False)
                qf1_next_target, qf2_next_target, _ = self.critic_target[agent_id].forward(next_state_batch, next_state_action)
                min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - self.alpha * next_state_log_pi
                next_q = reward_batch + self.gamma * (min_qf_next_target) * (1 - done_batch)
            next_q = next_q.detach()
            qf_1, qf_2, _ =  self.critic[agent_id].forward(state_batch, action_batch)  # Two Q-functions to mitigate positive bias in the policy improvement step
            qf1_loss = F.mse_loss(qf_1, next_q)
            qf2_loss = F.mse_loss(qf_2, next_q)
            total_loss = qf1_loss + qf2_loss
            self.critic_optim[agent_id]
            total_loss.backward()
            self.critic_optim[agent_id].step()
            ################ Policy Update ###########
            pi, log_pi, _, _, _ = self.policies[agent_id].noisy_action(state_batch, return_only_action=False)
            qf1_pi, qf2_pi, _ = self.critic[agent_id].forward(state_batch, pi)
            min_qf_pi = torch.min(qf1_pi, qf2_pi)
            policy_loss = ((self.alpha * log_pi) - min_qf_pi).mean()
            self.policy_optim[agent_id].zero_grad()
            policy_loss.backward()
            self.policy_optim[agent_id].step()

            self.num_updates += 1
            soft_update(self.critic_target[agent_id], self.critic[agent_id], self.tau)



    # Save model parameters
    def save_model(self, env_name, suffix="", actor_path=None, critic_path=None):
        if not os.path.exists('models/'):
            os.makedirs('models/')

        if actor_path is None:
            actor_path = "models/sac_actor_{}_{}".format(env_name, suffix)
        if critic_path is None:
            critic_path = "models/sac_critic_{}_{}".format(env_name, suffix)
        print('Saving models to {} and {}'.format(actor_path, critic_path))
        torch.save(self.policy.state_dict(), actor_path)
        torch.save(self.critic.state_dict(), critic_path)

    # Load model parameters
    def load_model(self, actor_path, critic_path):
        print('Loading models from {} and {}'.format(actor_path, critic_path))
        if actor_path is not None:
            self.policy.load_state_dict(torch.load(actor_path))
        if critic_path is not None:
            self.critic.load_state_dict(torch.load(critic_path))
