
#!/bin/bash

# to kill all the running jobs: pkill -f "python launch.py"

# Usage check
if [ "$#" -lt 2 ]; then
  echo "Usage: $0 <description_of_run> <config> [additional_params...]"
  exit 1
fi

# First argument: Description of the run
description="$1"
config_name="$2"
shift  # shift the arguments to the left so additional params can be accessed normally
shift

# Prepare the experiment root directory name
current_datetime=$(date "+%Y%m%d_%H%M%S")
no_space_description=$(echo "$description" | tr ' ' '_')
exp_root_dir="outputs/mass_runs/${current_datetime}_${no_space_description}"
mkdir -p "$exp_root_dir"

# List of available GPUs
declare -a gpus=(0 1 2 3 4 5 6 7)  # Update this array based on available GPUs

# Array of prompts; comment out prompts for GPUs that are occupied or not needed
declare -a prompts=(
  "A zoomed out DSLR photo of a hamburger"                           # GPU 0
  "A DSLR photograph of a soccer ball, 80mm"                         # GPU 1
  "A DSLR photo of a white fluffy cat"                               # GPU 2
  "A DSLR portrait of a beutiful woman with elf ears"                # GPU 3
  "A DSLR photo of an old man"                                       # GPU 4
  "A beautiful portrait photography of a black woman, 50 years old, short tousled brown hair, 3 point lighting, flash with softbox, by Annie Leibovitz, 80mm, hasselblad"                                          # GPU 5
  "Michelangelo style statue of dog reading news on a cellphone"     # GPU 6
  "A DSLR photograph of a coke can, 80mm"                            # GPU 7
)

# Additional parameters passed to the script (if any)
additional_params="$@"

# Check the length of gpus and prompts should be same
if [ ${#gpus[@]} -ne ${#prompts[@]} ]; then
  echo "Error: The number of GPUs does not match the number of prompts."
  exit 1
fi

# Loop through the available GPUs and launch jobs
for i in "${!gpus[@]}"
do
  gpu_index=${gpus[$i]}
  prompt_name=$(echo "${prompts[$i]}" | tr ' ' '_')
  if [ -n "${prompts[$i]}" ]; then  # Only run jobs for non-empty prompts
    output_file="${exp_root_dir}/${gpu_index}_${prompt_name}.txt"
    echo "Starting job on GPU $gpu_index with prompt: ${prompts[$i]}"
    (time python launch.py --config "configs/$config_name" --train --gpu "$gpu_index" \
         system.prompt_processor.prompt="${prompts[$i]}" \
         exp_root_dir="$exp_root_dir" \
         $additional_params) 2>&1 | tee "$output_file" &
    pids+=($!)  # Store PID of each job
  fi
done

echo "All applicable jobs have been started."

# Wait for all started jobs to finish
for pid in ${pids[@]}; do
    wait $pid
done

./mass_run_report.sh $exp_root_dir/ours-inversion

echo "All jobs have finished."
