from PIL import Image

def crop_sub_image(image_path, image_index, opacity_index=None, save_name="output.png"):
    """
    Crops a sub-image from a horizontally concatenated series of square images and optionally adds an opacity map from another sub-image.
    
    Args:
    image_path (str): The file path of the source image.
    image_index (int): The index of the sub-image to crop (0-indexed).
    opacity_index (int, optional): The index of the sub-image to use as an opacity map (0-indexed).

    Returns:
    None: Saves the cropped image as 'output.png' in the current directory.
    """
    try:
        # Open the image file
        with Image.open(image_path) as img:
            width, height = img.size
            num_images = width // height  # Calculate how many square images are there
            if image_index < 0 or image_index >= num_images:
                raise ValueError("Image index out of valid range.")
            if opacity_index is not None and (opacity_index < 0 or opacity_index >= num_images):
                raise ValueError("Opacity index out of valid range.")

            # Calculate the area to crop for the main image
            left = image_index * height
            upper = 0
            right = left + height
            lower = height

            # Crop the main image
            cropped_img = img.crop((left, upper, right, lower))

            if opacity_index is not None:
                # Calculate the area to crop for the opacity map
                left_opacity = opacity_index * height
                right_opacity = left_opacity + height

                # Crop the opacity map
                opacity_map = img.crop((left_opacity, upper, right_opacity, lower))
                opacity_map = opacity_map.convert('L')  # Convert to grayscale for alpha channel

                # Add the alpha channel to the main image
                cropped_img.putalpha(opacity_map)

            # Save the cropped image
            cropped_img.save(save_name)
            print(f"Image has been saved as {save_name}.")

    except FileNotFoundError:
        print("The specified file does not exist.")
    except ValueError as e:
        print(e)
    except Exception as e:
        print(f"An error occurred: {e}")

# Call the function with CLI arguments
if __name__ == '__main__':
    import sys
    if len(sys.argv) not in [3, 4]:
        print("Usage: python crop_subimg.py <image_path> <image_index> [opacity_index]")
    else:
        image_path = sys.argv[1]
        image_index = int(sys.argv[2])
        opacity_index = int(sys.argv[3]) if len(sys.argv) == 4 else None
        crop_sub_image(image_path, image_index, opacity_index)
