#!/bin/bash

# Configuration
gap_ratio=0.05                          # Gap ratio
subimage_idx=(3 0 4)                   # Subimage indices
it_idx=(500 500 1000 2000 3500 5000 8000)  # Iteration indices for the images
num_images=6                           # Number of images to process

# Ensure correct number of indices in it_idx
if [ ${#it_idx[@]} -lt $num_images ]; then
    echo "Error: The number of iteration indices is less than the required number of images."
    exit 1
fi

directory=$1  # The directory containing images

# Validate input
if [ -z "$directory" ]; then
    echo "Usage: $0 <directory>"
    exit 1
fi

temp_dir=$(mktemp -d) # Temporary directory for intermediate images

# Function to crop and save sub-images
process_image() {
    local idx=$1
    local file_name="${directory}/save/it${idx}-0.png"
    local output_file="${temp_dir}/result-${idx}.png"

    # Get the image width and height
    read img_width img_height < <(identify -format "%w %h" "$file_name")

    # Width of each sub-image
    local sub_img_width=$((img_width / ${#subimage_idx[@]}))

    # Prepare an array for holding sub-image files
    local sub_images=()

    # Loop over sub-indices, crop and save each sub-image
    for sub_idx in "${subimage_idx[@]}"; do
        local sub_image="${temp_dir}/sub-${idx}-${sub_idx}.png"
        local offset=$((sub_idx * sub_img_width))
        convert "$file_name" -crop "${sub_img_width}x${img_height}+${offset}+0" "$sub_image"
        sub_images+=("$sub_image")
    done

    # Calculate gap in pixels
    local gap_pixels=$(echo "$img_height * $gap_ratio" | bc)
    local gap_image="${temp_dir}/gap-${idx}.png"
    convert -size "${gap_pixels}x$img_height" xc:transparent "$gap_image"

    # Concatenate the sub-images vertically with gap
    convert "${sub_images[@]}" -append "$gap_image" "$output_file"
}

# Process images based on predefined indices
for idx in "${it_idx[@]:0:$num_images}"; do
    process_image "$idx"
done

# Concatenate all results horizontally
final_output="${directory}/final_result.png"
convert "${temp_dir}/result-"*.png +append "$final_output"

# Clean up temporary directory
rm -rf "$temp_dir"

echo "Final image created at $final_output"
