#!/bin/bash

# Usage check
if [ "$#" -lt 3 ]; then
  echo "Usage: $0 <description_of_run> <config> <prompt_file> [additional_params...]"
  exit 1
fi

description="$1"
config_name="$2"
prompt_file="$3"
shift 3  # shift the arguments to the left so additional params can be accessed normally

# Prepare the experiment root directory name
current_datetime=$(date "+%Y%m%d_%H%M%S")
no_space_description=$(echo "$description" | tr ' ' '_')
exp_root_dir="outputs/full_runs/${current_datetime}_${no_space_description}"
mkdir -p "$exp_root_dir"

# List of available GPUs
declare -a gpus=(0 1 2 3 4 5) # 4 5 6 7

# Read prompts from the file, ignoring lines that start with '#'
mapfile -t prompts < <(grep -v '^#' "$prompt_file")

# Check if the number of available GPUs is less than the number of prompts
if [ ${#gpus[@]} -lt ${#prompts[@]} ]; then
  echo "Warning: There are more prompts than GPUs. Prompts will be scheduled as GPUs become available."
fi

# Additional parameters passed to the script (if any)
additional_params="$@"

declare -A pids  # Associative array to hold process IDs indexed by GPU

# Function to check running jobs and find available GPUs
function find_available_gpus() {
  local available_gpus=()
  for gpu in "${gpus[@]}"; do
    if [[ -z "${pids[$gpu]}" ]] || ! kill -0 ${pids[$gpu]} 2>/dev/null; then
      # GPU is available if no PID is recorded or if the process is not running
      available_gpus+=("$gpu")
    fi
  done
  echo "${available_gpus[@]}"
}

# Main loop for assigning and managing jobs on GPUs
index=0
while [ $index -lt ${#prompts[@]} ]; do
  read -ra available_gpus <<< $(find_available_gpus)
  for gpu in "${available_gpus[@]}"; do
    if [ $index -ge ${#prompts[@]} ]; then
      break 2
    fi
    prompt="${prompts[$index]}"
    prompt_name=$(echo "${prompt}" | tr ' ' '_')
    output_file="${exp_root_dir}/${index}_${prompt_name}.txt"
    echo "Starting job on GPU $gpu with prompt: $prompt"
    (time python launch.py --config "$config_name" --train --gpu "$gpu" \
         system.prompt_processor.prompt="$prompt" \
         exp_root_dir="$exp_root_dir" \
         $additional_params) 2>&1 | tee "$output_file" &
    pids[$gpu]=$!  # Store PID of the job indexed by GPU
    ((index++))
  done
  sleep 5 # wait a bit before checking again to allow processes to start
done

echo "All jobs have been started. Waiting for them to finish..."

# Wait for all started jobs to finish
for pid in "${pids[@]}"; do
    wait $pid
done

echo "All jobs have finished."
