#!/bin/bash

# Usage check
if [ "$#" -ne 2 ]; then
    echo "Usage: $0 <root_path> <iteration_number>"
    exit 1
fi

root_path=$1
iteration=$2

# Prepare file paths
full_output="${root_path}/it${iteration}-full.mp4"
preview_output="${root_path}/it${iteration}-preview.mp4"
tempfile_full="/tmp/full_list_${iteration}.txt"
tempfile_preview="/tmp/preview_list_${iteration}.txt"

# Find video files and prepare file list for ffmpeg
# Using 'sort -V' for natural sort of (version) numbers within text
find "${root_path}" -type f -name "it${iteration}-test.mp4" | sort > $tempfile_full

# Check if files were found
if [ ! -s $tempfile_full ]; then
    echo "No video files found for iteration ${iteration}"
    exit 1
fi

# Format file list correctly for ffmpeg (this step may not be necessary if paths are used directly in the script)
sed -i "s|^|file '|; s|$|'|" $tempfile_full

####################################################################################
############# Concatenate all videos vertically for the full output ################
####################################################################################
# Initialize variables for FFmpeg's filter_complex
filter_complex=""
input_args=()
count=0
target_width=1280  # Desired width for all videos to match

# Read each line in the tempfile_full
while IFS= read -r line
do
    # Extract the filename safely by cutting on single quotes
    filename="$(echo "$line" | cut -d "'" -f 2)"

    # Append the input file option to the inputs array
    input_args+=("-i" "$filename")

    # Construct the filter_complex part for each video stream to scale videos maintaining their aspect ratio
    filter_complex+="[${count}:v]scale=$target_width:-2,setsar=1[v$count]; "

    # Increment the count of inputs
    count=$((count+1))
done < "$tempfile_full"

# Construct the vertical stacking part of the filter_complex
vstack_inputs=""
for ((i=0; i<count; i++)); do
    vstack_inputs+="[v$i]"
done
vstack_inputs+="vstack=inputs=$count[out]"

# Combine all parts of the filter_complex
filter_complex+="$vstack_inputs"

# Run FFmpeg with all inputs and the complete filter complex
ffmpeg "${input_args[@]}" -y -filter_complex "$filter_complex" -map "[out]" -c:v libx264 -crf 23 -preset fast $full_output
echo "Full output saved to: $full_output"

####################################################################################
########## Crop videos to left third and concatenate into a 4x2 grid ################
####################################################################################
# Reset the filter_complex for new processing
filter_complex=""
input_args=()
count=0

# Read each line in the tempfile_full again for the preview processing
while IFS= read -r line
do
    filename="$(echo "$line" | cut -d "'" -f 2)"
    input_args+=("-i" "$filename")
    crop_width=$((target_width / 3))
    filter_complex+="[${count}:v]scale=$target_width:-2,crop=$crop_width:ih:0:0,setsar=1[v$count]; "
    count=$((count+1))
done < "$tempfile_full"

# Check if the number of videos is not 8, exit or warn as necessary
if [ "$count" -ne 8 ]; then
    echo "Warning: The number of videos is not exactly 8, the output grid might not be 4x2 as expected."
    # Uncomment the next line to make the script exit if not exactly 8 videos
    # exit 1
fi

# Construct the grid stacking part of the filter_complex using xstack
xstack_inputs=""
for ((i=0; i<count; i++)); do
    xstack_inputs+="[v$i] "
done

xstack_inputs+="xstack=inputs=$count:layout=0_0|w0_0|0_h0|w0_h0|w0+w1_0|w0+w1_h0|w0+w1+w2_0|w0+w1+w2_h0[out]"

# Combine all parts of the filter_complex for the grid layout
filter_complex+="$xstack_inputs"

# Run FFmpeg with all inputs and the complete filter complex for the grid layout
ffmpeg "${input_args[@]}" -y -filter_complex "$filter_complex" -map "[out]" -c:v libx264 -crf 23 -preset fast $preview_output
echo "Grid output saved to: $preview_output"
