#!/usr/bin/env python
# coding=utf-8
# Copyright 2020 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" Finetuning the library models for sequence classification on GLUE."""
# You can also adapt this script on your own text classification task. Pointers for this are left as comments.

import argparse
import dataclasses
import json
import logging
import os
from pathlib import Path
import random
import sys
from dataclasses import dataclass, field
from typing import Optional

import datasets
import numpy as np
from datasets import load_dataset, load_metric
from examples_text_classification.util import DataLoader
import transformers
from transformers import AdamW
from transformers import (
    AutoConfig,
    AutoModelForSequenceClassification,
    AutoTokenizer,
    DataCollatorWithPadding,
    EvalPrediction,
    HfArgumentParser,
    PretrainedConfig,
    # Trainer,
    TrainingArguments,
    default_data_collator,
    set_seed,
)
from examples_text_classification.new_trainer import NewTextClassificationTrainer as Trainer

from transformers.trainer_utils import get_last_checkpoint
from transformers.utils.versions import require_version
from transformers import TrainingArguments
from torch.optim import LBFGS
# from torch.optim import SdLBFGS
from pca import PCA
from kfac import KFAC
from sgdw import SGDW
from adamsq import AdamSq
import torch
import os
import sys
from eigen_util import PyHessianCallback
from gpu_memory_monitor  import GPUMemoryMonitor

# os.environ["WANDB_DISABLED"] = "true"


# Will error if the minimal version of Transformers is not installed. Remove at your own risks.
# check_min_version("4.16.0.dev0")


require_version("datasets>=1.8.0", "To fix: pip install -r examples/pytorch/text-classification/requirements.txt")

task_to_keys = {
    "cola": ("sentence", None),
    "mnli": ("premise", "hypothesis"),
    "mrpc": ("sentence1", "sentence2"),
    "qnli": ("question", "sentence"),
    "qqp": ("question1", "question2"),
    "rte": ("sentence1", "sentence2"),
    "sst2": ("sentence", None),
    "stsb": ("sentence1", "sentence2"),
    "wnli": ("sentence1", "sentence2"),
    "superglue-boolq": ("question", "passage")
}


logger = logging.getLogger(__name__)

@dataclass
class TrainingArguments(TrainingArguments):
    # print_num_parameters: Optional[bool] = field(default=False, metadata={"help": "If set, print the parameters of "
                                                                                #  "the model."})
    do_test: Optional[bool] = field(default=False, metadata={"help": "If set, evaluates the test performance."})
    optimizer: Optional[str] = field(default=None, metadata={"help": "['pca', 'lbfgs']"})
    # split_validation_test: Optional[bool] = field(default=False,
                                                #   metadata={"help": "If set, for the datasets which do not"
                                                                    # "have the test set, we use validation set as their"
                                                                    # "test set and make a validation set from either"
                                                                    # "splitting the validation set into half (for smaller"
                                                                    # "than 10K samples datasets), or by using 1K examples"
                                                                    # "from training set as validation set (for larger"
                                                                    # " datasets)."})
    pyhessian: Optional[bool] = field(default=False)

@dataclass
class DataTrainingArguments:
    """
    Arguments pertaining to what data we are going to input our model for training and eval.

    Using `HfArgumentParser` we can turn this class
    into argparse arguments to be able to specify them on
    the command line.
    """

    task_name: Optional[str] = field(
        default=None,
        metadata={"help": "The name of the task to train on: " + ", ".join(task_to_keys.keys())},
    )
    dataset_name: Optional[str] = field(
        default=None, metadata={"help": "The name of the dataset to use (via the datasets library)."}
    )
    dataset_config_name: Optional[str] = field(
        default=None, metadata={"help": "The configuration name of the dataset to use (via the datasets library)."}
    )
    max_seq_length: int = field(
        default=128,
        metadata={
            "help": "The maximum total input sequence length after tokenization. Sequences longer "
            "than this will be truncated, sequences shorter will be padded."
        },
    )
    overwrite_cache: bool = field(
        default=False, metadata={"help": "Overwrite the cached preprocessed datasets or not."}
    )
    pad_to_max_length: bool = field(
        default=True,
        metadata={
            "help": "Whether to pad all samples to `max_seq_length`. "
            "If False, will pad the samples dynamically when batching to the maximum length in the batch."
        },
    )
    max_train_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of training examples to this "
            "value if set."
        },
    )
    max_eval_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of evaluation examples to this "
            "value if set."
        },
    )
    max_predict_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of prediction examples to this "
            "value if set."
        },
    )
    max_test_samples: Optional[int] = field(
        default=None,
        metadata={"help": "For debugging purposes or quicker training, truncate the number of test examples to this "
            "value if set."}
    )
    train_file: Optional[str] = field(
        default=None, metadata={"help": "A csv or a json file containing the training data."}
    )
    validation_file: Optional[str] = field(
        default=None, metadata={"help": "A csv or a json file containing the validation data."}
    )
    test_file: Optional[str] = field(default=None, metadata={"help": "A csv or a json file containing the test data."})

    def __post_init__(self):
        if self.task_name is not None:
            self.task_name = self.task_name.lower()
            if self.task_name not in task_to_keys.keys():
                raise ValueError("Unknown task, you should pick one in " + ",".join(task_to_keys.keys()))
        elif self.dataset_name is not None:
            pass
        elif self.train_file is None or self.validation_file is None:
            raise ValueError("Need either a GLUE task, a training/validation file or a dataset name.")
        else:
            train_extension = self.train_file.split(".")[-1]
            assert train_extension in ["csv", "json"], "`train_file` should be a csv or a json file."
            validation_extension = self.validation_file.split(".")[-1]
            assert (
                validation_extension == train_extension
            ), "`validation_file` should have the same extension (csv or json) as `train_file`."


@dataclass
class ModelArguments:
    """
    Arguments pertaining to which model/config/tokenizer we are going to fine-tune from.
    """

    model_name_or_path: str = field(
        metadata={"help": "Path to pretrained model or model identifier from huggingface.co/models"}
    )
    config_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained config name or path if not the same as model_name"}
    )
    tokenizer_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained tokenizer name or path if not the same as model_name"}
    )
    cache_dir: Optional[str] = field(
        default=None,
        metadata={"help": "Where do you want to store the pretrained models downloaded from huggingface.co"},
    )
    use_fast_tokenizer: bool = field(
        default=True,
        metadata={"help": "Whether to use one of the fast tokenizer (backed by the tokenizers library) or not."},
    )
    model_revision: str = field(
        default="main",
        metadata={"help": "The specific model version to use (can be a branch name, tag name or commit id)."},
    )
    use_auth_token: bool = field(
        default=False,
        metadata={
            "help": "Will use the token generated when running `transformers-cli login` (necessary to use this script "
            "with private models)."
        },
    )


class RemainArgHfArgumentParser(HfArgumentParser):
    def parse_json_file(self, json_file: str, return_remaining_args=True, local_rank=-1):
        """
        Alternative helper method that does not use `argparse` at all, instead loading a json file and populating the
        dataclass types.
        """
        data = json.loads(Path(json_file).read_text())
        data["local_rank"] = local_rank
        outputs = []
        for dtype in self.dataclass_types:
            keys = {f.name for f in dataclasses.fields(dtype) if f.init}
            inputs = {k: data.pop(k) for k in list(data.keys()) if k in keys}
            obj = dtype(**inputs)
            outputs.append(obj)
        
        remain_args = argparse.ArgumentParser()
        remain_args.__dict__.update(data)
        if return_remaining_args:
            return (*outputs, remain_args)
        else:
            return (*outputs,)


def main():
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.
    main_parser = argparse.ArgumentParser()
    main_parser.add_argument("--config", default=None, type=str)
    main_parser.add_argument("--lr", default=1e-4, type=float)
    # for PCA
    main_parser.add_argument("--damp2", default=1e-2, type=float)
    main_parser.add_argument("--alpha", default=0.1, type=float)
    # for SGD
    main_parser.add_argument("--momentum", default=0.9, type=float)
    main_parser.add_argument("--weight_decay", default=0.0, type=float)
    # main_parser.add_argument("--local_rank", default=0, type=int)
    # for kfac
    main_parser.add_argument("--gamma", default=1.0, type=float)
    main_parser.add_argument("--update_freq", default=1, type=int)
    #for adamsq
    main_parser.add_argument("--eps", default=0.05, type=float)
    # batch size (will replace the original batch size in config file)
    main_parser.add_argument("--per_device_train_batch_size", default=100, type=int)
    main_parser.add_argument("--per_device_eval_batch_size", default=100, type=int)
    main_parser.add_argument("--gradient_accumulation_steps", default=1, type=int)
    main_parser.add_argument("--max_grad_norm", default=0.1, type=float)
    main_parser.add_argument("--pre_clipping", default=1, type=int)


    main_parser.add_argument("--pyhessian", default=False, action="store_true")

    main_parser.add_argument("--seed", default=42, type=int)
    main_parser.add_argument("--std", default=-1, type=float)
    main_parser.add_argument("--lora_rank", default=8, type=int)

    main_args = main_parser.parse_args()

    if "LOCAL_RANK" in os.environ:
        local_rank = int(os.environ["LOCAL_RANK"])
    else:
        local_rank = -1

    parser = RemainArgHfArgumentParser((ModelArguments, DataTrainingArguments, TrainingArguments))
    # if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
    if main_args.config:
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        # json_file=os.path.abspath(sys.argv[1])
        json_file = os.path.abspath(main_args.config)
        model_args, data_args, training_args, delta_args = parser.parse_json_file(json_file, return_remaining_args=True, local_rank=local_rank) #args = arg_string, return_remaining_strings=True) #parse_json_file(json_file=os.path.abspath(sys.argv[1]))
        print("done")
    elif len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        json_file = os.path.abspath(sys.argv[1])
        model_args, data_args, training_args, delta_args = parser.parse_json_file(json_file, return_remaining_args=True, local_rank=local_rank)
    else:
        model_args, data_args, training_args, delta_args = parser.parse_args_into_dataclasses()


    training_args.per_device_train_batch_size = main_args.per_device_train_batch_size
    training_args.per_device_eval_batch_size = main_args.per_device_eval_batch_size
    delta_args.lora_r = main_args.lora_rank
    
    #training_args.optimizer = main_args.optim
    if training_args.optimizer in ['kfac', 'kfacw']:
        training_args.output_dir = training_args.output_dir + "/bs_"+ str(training_args.per_device_train_batch_size*training_args.world_size) + "_lr_" + str(main_args.lr) + "_gamma_" + str(main_args.gamma) + "_update_freq_" + str(main_args.update_freq)\
        + "_momentum_" + str(main_args.momentum) + "_weight_decay_" + str(main_args.weight_decay) + "_eps_" + str(main_args.eps) + "_max_grad_norm_" + str(main_args.max_grad_norm) + "_pre_clipping_" + str(main_args.pre_clipping)
        training_args.run_name = "-".join([training_args.run_name, "bs_"+str(training_args.per_device_train_batch_size*training_args.world_size), "lr_"+str(main_args.lr), 
        "gamma_" + str(main_args.gamma), "update_freq_" + str(main_args.update_freq), "momentum_" + str(main_args.momentum), "weight_decay_" + str(main_args.weight_decay),
         "eps_" + str(main_args.eps), "max_grad_norm_" + str(main_args.max_grad_norm), "pre_clipping_" + str(main_args.pre_clipping), "seed_" + str(main_args.seed)])  
    elif training_args.optimizer in ['adamw', None]:
        training_args.output_dir = training_args.output_dir + "/bs_"+ str(training_args.per_device_train_batch_size*training_args.world_size) + "_lr_" + str(main_args.lr) + "_weight_decay_" + str(main_args.weight_decay)\
        + "_max_grad_norm_"+ str(main_args.max_grad_norm) + "_pre_clipping_" + str(main_args.pre_clipping)
        training_args.run_name = "-".join([training_args.run_name, "bs_"+str(training_args.per_device_train_batch_size*training_args.world_size), "lr_" + str(main_args.lr), "weight_decay_" + str(main_args.weight_decay),
        "max_grad_norm_" + str(main_args.max_grad_norm), "pre_clipping_" + str(main_args.pre_clipping), "seed_" + str(main_args.seed)])
    elif training_args.optimizer =='adamsq':
        training_args.output_dir = training_args.output_dir + "/bs_"+ str(training_args.per_device_train_batch_size*training_args.world_size) + "_lr_" + str(main_args.lr) + "_weight_decay_" + str(main_args.weight_decay) + "_eps_" + str(main_args.eps)
    else:
        pass
    training_args.pyhessian = main_args.pyhessian
    training_args.learning_rate = main_args.lr
    training_args.gradient_accumulation_steps = main_args.gradient_accumulation_steps
    training_args.max_grad_norm = main_args.max_grad_norm
    training_args.pre_clipping = main_args.pre_clipping

    #if training_args.report_to == "wandb":
    import wandb
    os.environ["WANDB_API_KEY"] = ""
    os.environ["WANDB_MODE"] = "offline"
    if os.getenv("WANDB_PROJECT", None) is None:
        os.environ["WANDB_PROJECT"] = "test-project"
    if os.getenv("WANDB_ENTITY", None) is None:
        os.environ["WANDB_ENTITY"] = "2nd-order-optim"
    if local_rank == 0 or local_rank == -1:
        wandb.login()
        print("initializing wandb")
        wandb.init()
        wandb.run.name = training_args.run_name

    # os.environ["WANDB_API_KEY"] = "9eb5742991d6d884b13810ff34e85db8dd6e5788"
    # os.environ["WANDB_MODE"] = "offline"
    # wandb.login()
    # wandb.init(project=os.getenv("WANDB_PROJECT", "test-project"), entity="2nd-order-optim-initialization", settings=wandb.Settings(_disable_stats=True))
    # wandb.run.name = training_args.run_name
    
    # Setup logging
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )

    log_level = training_args.get_process_log_level()
    logger.setLevel(log_level)
    datasets.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.enable_default_handler()
    transformers.utils.logging.enable_explicit_format()
    # Log on each process the small summary:
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f"distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    logger.info(f"Training/evaluation parameters {training_args}")

    # Detecting last checkpoint.
    last_checkpoint = None
    if os.path.isdir(training_args.output_dir) and not training_args.overwrite_output_dir:
        last_checkpoint = get_last_checkpoint(training_args.output_dir)
        if last_checkpoint is None and len(os.listdir(training_args.output_dir)) > 0:
            raise ValueError(
                f"Output directory ({training_args.output_dir}) already exists and is not empty. "
                "Use --overwrite_output_dir to overcome."
            )
        elif last_checkpoint is not None and training_args.resume_from_checkpoint is None:
            logger.info(
                f"Checkpoint detected, resuming training at {last_checkpoint}. To avoid this behavior, change "
                "the `--output_dir` or add `--overwrite_output_dir` to train from scratch."
            )

    # Set seed before initializing model.
    training_args.seed = main_args.seed
    print(f"Using random seed {training_args.seed}")
    set_seed(training_args.seed)

    # Get the datasets: you can either provide your own CSV/JSON training and evaluation files (see below)
    # or specify a GLUE benchmark task (the dataset will be downloaded automatically from the datasets Hub).
    #
    # For CSV/JSON files, this script will use as labels the column called 'label' and as pair of sentences the
    # sentences in columns called 'sentence1' and 'sentence2' if such column exists or the first two columns not named
    # label if at least two columns are provided.
    #
    # If the CSVs/JSONs contain only one non-label column, the script does single sentence classification on this
    # single column. You can easily tweak this behavior (see below)
    #
    # In distributed training, the load_dataset function guarantee that only one local process can concurrently
    # download the dataset.
    if data_args.task_name is not None:
        # Downloading and loading a dataset from the hub.
        data_hub_name = "glue"
        task_name = data_args.task_name
        if data_args.task_name.startswith("superglue"):
            data_hub_name = "super_glue"
            task_name = data_args.task_name.split("-")[-1]
        raw_datasets = load_dataset(data_hub_name, task_name, cache_dir=model_args.cache_dir, script_version="master")
        # load data offline
        # from datasets import load_from_disk
        # raw_datasets = load_from_disk(f"/path/to/dataset/{data_args.task_name}")
    
    elif data_args.dataset_name is not None:
        # Downloading and loading a dataset from the hub.
        raw_datasets = load_dataset(
            data_args.dataset_name, data_args.dataset_config_name, cache_dir=model_args.cache_dir
        )
    else:
        # Loading a dataset from your local files.
        # CSV/JSON training and evaluation files are needed.
        data_files = {"train": data_args.train_file, "validation": data_args.validation_file}

        # Get the test dataset: you can provide your own CSV/JSON test file (see below)
        # when you use `do_predict` without specifying a GLUE benchmark task.
        if training_args.do_predict:
            if data_args.test_file is not None:
                train_extension = data_args.train_file.split(".")[-1]
                test_extension = data_args.test_file.split(".")[-1]
                assert (
                    test_extension == train_extension
                ), "`test_file` should have the same extension (csv or json) as `train_file`."
                data_files["test"] = data_args.test_file
            else:
                raise ValueError("Need either a GLUE task or a test file for `do_predict`.")

        for key in data_files.keys():
            logger.info(f"load a local file for {key}: {data_files[key]}")

        if data_args.train_file.endswith(".csv"):
            # Loading a dataset from local csv files
            raw_datasets = load_dataset("csv", data_files=data_files, cache_dir=model_args.cache_dir)
        else:
            # Loading a dataset from local json files
            raw_datasets = load_dataset("json", data_files=data_files, cache_dir=model_args.cache_dir)
    # See more about loading any type of standard or custom dataset at
    # https://huggingface.co/docs/datasets/loading_datasets.html.

    # Labels
    if data_args.task_name is not None:
        is_regression = data_args.task_name == "stsb"
        if not is_regression:
            label_list = raw_datasets["train"].features["label"].names
            num_labels = len(label_list)
        else:
            num_labels = 1
    else:
        # Trying to have good defaults here, don't hesitate to tweak to your needs.
        is_regression = raw_datasets["train"].features["label"].dtype in ["float32", "float64"]
        if is_regression:
            num_labels = 1
        else:
            # A useful fast method:
            # https://huggingface.co/docs/datasets/package_reference/main_classes.html#datasets.Dataset.unique
            label_list = raw_datasets["train"].unique("label")
            label_list.sort()  # Let's sort it for determinism
            num_labels = len(label_list)

    # Load pretrained model and tokenizer
    #
    # In distributed training, the .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.
    config = AutoConfig.from_pretrained(
        model_args.config_name if model_args.config_name else model_args.model_name_or_path,
        num_labels=num_labels,
        finetuning_task=data_args.task_name,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    tokenizer = AutoTokenizer.from_pretrained(
        model_args.tokenizer_name if model_args.tokenizer_name else model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        use_fast=model_args.use_fast_tokenizer,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    if tokenizer.pad_token is None:
        if tokenizer.eos_token is not None:
            tokenizer.pad_token = tokenizer.eos_token
            config.pad_token_id = config.eos_token_id # some models check pad_token_id in config, e.g. ctrl
        else:
            tokenizer.add_special_tokens({'pad_token': '[PAD]'})
            config.pad_token_id = config.vocab_size

    model = AutoModelForSequenceClassification.from_pretrained(
        model_args.model_name_or_path,
        from_tf=bool(".ckpt" in model_args.model_name_or_path),
        config=config,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )

    model.resize_token_embeddings(len(tokenizer))
    if delta_args.delta_type.lower() != "none":
        from opendelta import AutoDeltaConfig
        from opendelta.auto_delta import AutoDeltaModel
        delta_config = AutoDeltaConfig.from_dict(vars(delta_args))
        delta_model = AutoDeltaModel.from_config(delta_config, backbone_model=model)
        delta_model.freeze_module(set_state_dict = True)
        delta_model.log(delta_ratio=True, trainable_ratio=True, visualization=False)
    
    # if main_args.std >= 0:
    #     for name, param in model.named_parameters():
    #         if param.requires_grad:
    #             print(f'name:{name}')
    #             print(f'shape:{param.shape}')
    #             print(f'former norm:{torch.norm(param).item()}')
    #             param = main_args.std * torch.randn_like(param).cuda()
    #             print(f'new norm:{torch.norm(param).item()}\n')



    # Preprocessing the raw_datasets
    if data_args.task_name is not None:
        sentence1_key, sentence2_key = task_to_keys[data_args.task_name]
    else:
        # Again, we try to have some nice defaults but don't hesitate to tweak to your use case.
        non_label_column_names = [name for name in raw_datasets["train"].column_names if name != "label"]
        if "sentence1" in non_label_column_names and "sentence2" in non_label_column_names:
            sentence1_key, sentence2_key = "sentence1", "sentence2"
        else:
            if len(non_label_column_names) >= 2:
                sentence1_key, sentence2_key = non_label_column_names[:2]
            else:
                sentence1_key, sentence2_key = non_label_column_names[0], None

    # Padding strategy
    if data_args.pad_to_max_length:
        padding = "max_length"
    else:
        # We will pad later, dynamically at batch creation, to the max sequence length in each batch
        padding = False

    # Some models have set the order of the labels to use, so let's make sure we do use it.
    label_to_id = None
    if (
        model.config.label2id != PretrainedConfig(num_labels=num_labels).label2id
        and data_args.task_name is not None
        and not is_regression
    ):
        # Some have all caps in their config, some don't.
        label_name_to_id = {k.lower(): v for k, v in model.config.label2id.items()}
        if list(sorted(label_name_to_id.keys())) == list(sorted(label_list)):
            label_to_id = {i: int(label_name_to_id[label_list[i]]) for i in range(num_labels)}
        else:
            logger.warning(
                "Your model seems to have been trained with labels, but they don't match the dataset: ",
                f"model labels: {list(sorted(label_name_to_id.keys()))}, dataset labels: {list(sorted(label_list))}."
                "\nIgnoring the model labels as a result.",
            )
    elif data_args.task_name is None and not is_regression:
        label_to_id = {v: i for i, v in enumerate(label_list)}

    if label_to_id is not None:
        model.config.label2id = label_to_id
        model.config.id2label = {id: label for label, id in config.label2id.items()}
    elif data_args.task_name is not None and not is_regression:
        model.config.label2id = {l: i for i, l in enumerate(label_list)}
        model.config.id2label = {id: label for label, id in config.label2id.items()}

    if data_args.max_seq_length > tokenizer.model_max_length:
        logger.warning(
            f"The max_seq_length passed ({data_args.max_seq_length}) is larger than the maximum length for the"
            f"model ({tokenizer.model_max_length}). Using max_seq_length={tokenizer.model_max_length}."
        )
    max_seq_length = min(data_args.max_seq_length, tokenizer.model_max_length)

    def preprocess_function(examples):
        # Tokenize the texts
        args = (
            (examples[sentence1_key],) if sentence2_key is None else (examples[sentence1_key], examples[sentence2_key])
        )
        result = tokenizer(*args, padding=padding, max_length=max_seq_length, truncation=True)

        # Map labels to IDs (not necessary for GLUE tasks)
        if label_to_id is not None and "label" in examples:
            result["label"] = [(label_to_id[l] if l != -1 else -1) for l in examples["label"]]
        return result

    with training_args.main_process_first(desc="dataset map pre-processing"):
        raw_datasets = raw_datasets.map(
            preprocess_function,
            batched=True,
            load_from_cache_file=False,
            desc="Running tokenizer on dataset",
        )
    dataloader = DataLoader(raw_datasets, data_args, model_args, training_args)
    if training_args.do_train:
        if "train" not in raw_datasets:
            raise ValueError("--do_train requires a train dataset")
        
        train_dataset = dataloader.get(split='train')
        if data_args.max_train_samples is not None:
            print("sample training set")
            train_dataset = train_dataset.select(list(random.sample(range(len(train_dataset)), data_args.max_train_samples)))


    if training_args.do_eval:
        if "validation" not in raw_datasets and "validation_matched" not in raw_datasets:
            raise ValueError("--do_eval requires a validation dataset")

        # eval_dataset = raw_datasets["validation_matched" if data_args.task_name == "mnli" else "validation"]
        eval_dataset = dataloader.get(split='validation')
        if data_args.max_eval_samples is not None:
            eval_dataset = eval_dataset.select(list(random.sample(range(len(eval_dataset)), data_args.max_eval_samples)))
    
    if training_args.do_test or data_args.task_name is not None or data_args.test_file is not None:
        if "test" not in raw_datasets and "test_matched" not in raw_datasets and 'validation' not in raw_datasets and 'validation_matched' not in raw_datasets:
            raise ValueError("--do_test requires a test or validation dataset")
        test_dataset = dataloader.get(split='test')
        if data_args.max_test_samples is not None:
            test_dataset = test_dataset.select(list(random.sample(range(len(test_dataset)), data_args.max_test_samples)))

    # if training_args.do_predict or data_args.task_name is not None or data_args.test_file is not None:
    #     if "test" not in raw_datasets and "test_matched" not in raw_datasets:
    #         raise ValueError("--do_predict requires a test dataset")
    #     predict_dataset = raw_datasets["test_matched" if data_args.task_name == "mnli" else "test"]
    #     if data_args.max_predict_samples is not None:
    #         predict_dataset = predict_dataset.select(range(data_args.max_predict_samples))

    # Log a few random samples from the training set:
    if training_args.do_train:
        for index in random.sample(range(len(train_dataset)), 3):
            logger.info(f"Sample {index} of the training set: {train_dataset[index]}.")

    # Get the metric function
    if data_args.task_name is not None:
        # metric = load_metric("glue", data_args.task_name)
        metric = load_metric("./examples_text_classification/metrics/glue.py", data_args.task_name)
    else:
        metric = load_metric("accuracy")

    # You can define your custom compute_metrics function. It takes an `EvalPrediction` object (a namedtuple with a
    # predictions and label_ids field) and has to return a dictionary string to float.
    def compute_metrics(p: EvalPrediction):
        preds = p.predictions[0] if isinstance(p.predictions, tuple) else p.predictions
        preds = np.squeeze(preds) if is_regression else np.argmax(preds, axis=1)
        # print("pred:", preds[:10])
        # print("label:", p.label_ids[:10])
        if data_args.task_name is not None:
            result = metric.compute(predictions=preds, references=p.label_ids)
            if len(result) > 1:
                result["combined_score"] = np.mean(list(result.values())).item()
            return result
        elif is_regression:
            return {"mse": ((preds - p.label_ids) ** 2).mean().item()}
        else:
            return {"accuracy": (preds == p.label_ids).astype(np.float32).mean().item()}

    # Data collator will default to DataCollatorWithPadding, so we change it if we already did the padding.
    if data_args.pad_to_max_length:
        data_collator = default_data_collator
    elif training_args.fp16:
        data_collator = DataCollatorWithPadding(tokenizer, pad_to_multiple_of=8)
    else:
        data_collator = None
    

    # params = [p for p in model.parameters() if p.requires_grad]
    #print(params.type)
    optimizer = None
    # print("number of trainable parameters:", len([p for p in model.parameters() if p.requires_grad]))
    # sys.exit()
    if training_args.optimizer is not None:
        if training_args.optimizer == "pca":
            adamw_optim = AdamW([p for p in model.parameters() if p.requires_grad], lr = main_args.lr, weight_decay = main_args.weight_decay)
            optimizer = PCA(adamw_optim, damp2=main_args.damp2, alpha=main_args.alpha)
        elif training_args.optimizer == "lbfgs":
            optimizer = LBFGS([p for p in model.parameters() if p.requires_grad], lr=main_args.lr)
        elif training_args.optimizer in ["sgd", "kfac"]:
            print('use sgd as optim')
            #preconditioner = KFAC(model, eps=0.05, update_freq=500, distributed=False)
            #optimizer = AdamW([p for p in model.parameters() if p.requires_grad], lr = main_args.lr)
            optimizer = torch.optim.SGD([p for p in model.parameters() if p.requires_grad], lr=main_args.lr, momentum=main_args.momentum, weight_decay=main_args.weight_decay)
        elif training_args.optimizer in ["sgdw", "kfacw"]:
            print('use sgdw as optim')
            #preconditioner = KFAC(model, eps=0.05, update_freq=500, distributed=False)
            #optimizer = AdamW([p for p in model.parameters() if p.requires_grad], lr = main_args.lr)
            optimizer = SGDW([p for p in model.parameters() if p.requires_grad], lr=main_args.lr, momentum=main_args.momentum, weight_decay=main_args.weight_decay)
        elif training_args.optimizer == "adamsq":
            print('use adamsq as optim')
            optimizer = AdamSq([p for p in model.parameters() if p.requires_grad], lr=main_args.lr, weight_decay=main_args.weight_decay, eps=main_args.eps)
        elif training_args.optimizer == "adamw":
            # do nothing, transformers default to adamw
            pass
        else:
            raise NotImplementedError

    #  set kfac args
    training_args.update_freq = main_args.update_freq
    training_args.gamma = main_args.gamma
    training_args.eps = main_args.eps
    
    # Initialize our Trainer
    
    trainer = Trainer(
        model=model,
        args=training_args,
        train_dataset=train_dataset if training_args.do_train else None,
        eval_dataset=eval_dataset if training_args.do_eval else None,
        compute_metrics=compute_metrics,
        tokenizer=tokenizer,
        data_collator=data_collator,
        optimizers=(optimizer, None)
    )
    trainer.add_callback(PyHessianCallback()) # do pyhessian 

    trainer.use_amp = False

    if last_checkpoint:
        model.load_state_dict(torch.load(os.path.join(last_checkpoint, "pytorch_model.bin"), map_location="cpu"), strict=False)

    # Training
    if training_args.do_train:
        checkpoint = None
        if training_args.resume_from_checkpoint is not None:
            checkpoint = training_args.resume_from_checkpoint
        elif last_checkpoint is not None:
            checkpoint = last_checkpoint
        # start gpu monitor
        if local_rank == 0 or local_rank == -1:
            monitor = GPUMemoryMonitor(interval=10) # log gpu memory every 10 seconds
            monitor.start()
            train_result = trainer.train(resume_from_checkpoint=checkpoint)
            monitor.stop()
        else:
            train_result = trainer.train(resume_from_checkpoint=checkpoint)

        # stop monitor after training is finished
        if train_result:
            metrics = train_result.metrics
            max_train_samples = (
                data_args.max_train_samples if data_args.max_train_samples is not None else len(train_dataset)
            )
            metrics["train_samples"] = min(max_train_samples, len(train_dataset))

            trainer.save_model()  # Saves the tokenizer too for easy upload

            trainer.log_metrics("train", metrics)
            trainer.save_metrics("train", metrics)
        # trainer.save_state()

    results = {}
    # Evaluation
    if training_args.do_eval:
        logger.info("*** Evaluate ***")

        # Loop to handle MNLI double evaluation (matched, mis-matched)
        tasks = [data_args.task_name]
        eval_datasets = [eval_dataset]
        if data_args.task_name == "mnli":
            tasks.append("mnli-mm")
            eval_datasets.append(raw_datasets["validation_mismatched"])

        for eval_dataset, task in zip(eval_datasets, tasks):
            metrics = trainer.evaluate(eval_dataset=eval_dataset)

            max_eval_samples = (
                data_args.max_eval_samples if data_args.max_eval_samples is not None else len(eval_dataset)
            )
            metrics["eval_samples"] = min(max_eval_samples, len(eval_dataset))

            trainer.log_metrics("eval", metrics)
            trainer.save_metrics("eval", metrics)
        results['eval'] = metrics

    if training_args.do_test:
        logger.info("*** Test ***")
        metrics = trainer.evaluate(eval_dataset=test_dataset)
        trainer.log_metrics("test", metrics)
        trainer.save_metrics("test", metrics)
        results['test'] = metrics
    
    return results

def _mp_fn(index):
    # For xla_spawn (TPUs)
    main()


if __name__ == "__main__":
    results = main()
    print(results)

