from typing import Literal, Iterable
from ._base import Config, param
import abc


type Decode = Literal["sampling", "greedy"]
type Reflect = Literal[False, "self", "oracle", "random"]
type Execution = tuple[Decode, Reflect, bool]


class BaseTestingArgBuilder(abc.ABC):

    @abc.abstractmethod
    def arg(self, decode: Decode, reflect: Reflect, traceback: bool) -> Config.TestArgs:
        ...
    
    def key(self, decode: Decode, reflect: Reflect, traceback: bool):
        return '-'.join(filter(bool, [
            decode,
            f"{reflect}-reflect" if reflect else "no-reflect",
            "traceback" if traceback else "",
        ]))

    def _build(self, executions: Iterable[Execution]) -> dict[str | None, Config.TestArgs]:
        return {self.key(decode, reflect, traceback): self.arg(decode, reflect, traceback)
                for decode, reflect, traceback in executions}

    def build(self, reflective: bool = True):
        return self._build(self._executions(reflective))

    def _executions(self, reflective: bool = True) -> Iterable[Execution]:
        for decode in ("sampling", "greedy"):
            yield decode, False, False
            if reflective:
                for reflect in ("oracle", "self", "random"):
                    for traceback in (True, False):
                        yield decode, reflect, traceback
