import dataclasses as dc
from typing import Literal, Callable
import math
import numpy as np


ProgressFn = Callable[[float, float, float], float]

def _linear(start: float, delta: float, k: float):
    return start + k * delta

def _sin(start: float, delta: float, k: float):
    return start + math.sin(k * math.pi / 2) * delta

def _cos(start: float, delta: float, k: float):
    return start + (1 - math.cos(k * math.pi / 2)) * delta


_funcs: dict[str, ProgressFn] = {
    'linear': _linear,
    'sin': _sin,
    'cos': _cos,
}


@dc.dataclass
class Progress:

    _Supported = Literal['linear', 'cos', 'sin']

    start_value: float
    end_value: float
    start_progress: float | int = 0
    end_progress: float | int = 1
    func: _Supported = 'linear'

    def __post_init__(self):

        if not (self.start_progress <= self.end_progress):
            raise ValueError
        
        self.delta = self.end_value - self.start_value
        self.gap = self.end_progress - self.start_progress

    def __call__(self, progress: float | int) -> float:
        
        if progress <= self.start_progress:
            return self.start_value
        elif progress > self.end_progress:
            return self.end_value
        else:
            k = (progress - self.start_progress) / self.gap

        try:
            func = _funcs[self.func]
        except KeyError:
            raise ValueError(
                f"{self.func} is not a supported function. The supported are: " +
                ', '.join("\"%s\"" % k for k in  _funcs.keys())
            )

        return func(self.start_value, self.delta, k)

    def _debug_plot(self, start: float, stop: float, resolution: int = 1000):
        import matplotlib.pyplot as plt

        xs = np.linspace(start, stop, resolution)
        y = np.vectorize(self.__call__, otypes=[np.float32])(xs)
        plt.plot(xs, y)
        plt.show()
