import sys
from argparse import ArgumentParser
from dataclasses import dataclass, asdict
from typing import Any, Literal, cast, overload, Sequence, ClassVar
from inspect import cleandoc


@dataclass
class PArg[T: (str, int, float)]:
    type: type[T]
    help: str | None = None
    choices: tuple[T, ...] | None = None


@dataclass
class KArg[T: (str, int, float, None)]:
    type: type[T]
    help: str | None = None
    action: str | None = None
    default: T | None = None
    short: str | None = None
    choices: tuple[T, ...] | None = None
    required: bool = False


def parg[T: (str, int, float)](
    type: type[T],
    help: str | None = None,
    choices: tuple[T, ...] | None = None
) -> T:
    return cast(Any, PArg(type, help, choices))


@overload
def karg(
    type: type[bool],
    help: str | None = None,
    *,
    action: Literal["store_true", "store_false"] | None = None,
    short: str | None = None,
) -> bool: ...
@overload
def karg[T: (str, int, float)](
    type: type[T],
    help: str | None = None,
    *,
    short: str | None = None,
    choices: tuple[T, ...] | None = None,
    default: T,
) -> T: ...
@overload
def karg[T: (str, int, float)](
    type: type[T],
    help: str | None = None,
    *,
    short: str | None = None,
    choices: tuple[T, ...] | None = None,
    required: Literal[True],
) -> T: ...
@overload
def karg[T: (str, int, float)](
    type: type[T],
    help: str | None = None,
    *,
    short: str | None = None,
    choices: tuple[T, ...] | None = None,
) -> T | None: ...
def karg[T: (str, int, float, bool)](
    type: type[T],
    help: str | None = None,
    action: str | None = None,
    short: str | None = None,
    choices: tuple[T, ...] | None = None,
    default: T | None = None,
    required: bool = False,
) -> T | None:
    if issubclass(type, bool):
        action = action or "store_true"
    return cast(Any, KArg(type, help, action, default, short, choices, required))
    

class Command:

    description: str | None = None

    @classmethod
    def _make_parser(cls, **options):
        parser = ArgumentParser(**options)
        pargs = [(name, v) for name, v in cls.__dict__.items() if isinstance(v, PArg)]
        kargs = [(name, v) for name, v in cls.__dict__.items() if isinstance(v, KArg)]
        for name, arg in pargs:
            options = {k: v for k, v in asdict(arg).items() if v is not None}
            parser.add_argument(name.replace('_', '-'), **options)
        for name, arg in kargs:
            names = ["--" + name.replace('_', '-')]
            if arg.short:
                names.append("-" + arg.short)
            options = {k: v for k, v in asdict(arg).items() if v is not None}
            options.pop("short", None)
            if issubclass(arg.type, bool):
                options.pop("type", None)
            parser.add_argument(*names, **options)
        return parser
    
    def __init__(self, args: Sequence[str] | None = None, **parser_options):
        parser = self._make_parser(**parser_options)
        namespace = parser.parse_args(args)
        for name, value in vars(namespace).items():
            setattr(self, name, value)
        self.__post_init__()
    
    def __post_init__(self): ...
    
    def __repr__(self):
        return '(' + ", ".join("%s=%s" % (k, repr(v)) for k, v in self.__dict__.items()) + ')'

    def __call__(self): ...

    @classmethod
    def get_help_str(cls) -> str:
        if cls.description is not None:
            return cls.description
        elif cls.__doc__ is not None:
            return cleandoc(cls.__doc__)
        else:
            return "No description"


def run(_globals: dict[str, Any], /, argv: list[str] | None = None, helpcmd: str | None = "help"):
    """
    According to `argv` (by default, use `sys.argv[1:]`), choose a command class
    from `_globals`, instantiate the command, and then run `__call__` procedure.

    Define your commands:

    ```python
    from exprutils import run, Command, karg, parg

    # define your commands
    class MyCmd(Command):
        # define positional and keyword arguments
        dog = parg(...) 
        cat = karg(...)

        # define main procedure
        def __call__(self): ...

    if __name__ == "__main__":
        run(globals())
    ```

    Use your command by:

    ```
    python file.py mycmd DOG --cat CAT 
    ```

    if `help` is given, an additional `help` command is added to show descriptions of commands in `_globals`.
    """

    if argv is None:
        prog_name = sys.argv[0]
        argv = sys.argv[1:]
    else:
        prog_name = None
    
    cmd_name = argv[0].strip()
    if prog_name is not None:    
        prog_name = prog_name + " " + cmd_name
    else:
        prog_name = cmd_name
    
    commands = {k.lower(): v for k, v in _globals.items()
                if isinstance(v, type) and issubclass(v, Command) and v is not Command}

    if cmd_name == helpcmd:
        import pprint
        for k, cls in commands.items():
            print(f"# {k}")
            h = '| ' + cls.get_help_str()
            h = h.replace('\n', "\n| ")
            print(h)
        print("Run any command with `-h` or `--help` to further help.")
    try:
        cmd_cls = commands[cmd_name]
    except KeyError:
        supported = ', '.join(("\"%s\"" % k.removeprefix('do_')) for k in commands.keys())
        print(f"\"{cmd_name}\" is not a supported command to run. "
              f"The supported commands are {supported}.")
        exit()
    
    command = cmd_cls(argv[1:], prog=prog_name)
    command()
