import numpy as np
import json
import re
import matplotlib.pyplot as plt
from pathlib import Path
from typing import Callable, Mapping, Literal, Any, LiteralString, Sequence, Iterable
from .figutils import tabulate_results, get_eval_results, Result, Results, PathLike


type SudokuResults = Results[int]


def key2blanks(k: str):
    if m := re.match(r"(\d+) blanks", k):
        blanks = int(m.group(1))
        return blanks
    else:
        raise ValueError


def _plot_hist(results: SudokuResults,
               key: str,
               title: str | None,
               bound: int | None,
               title_fontsize: int,
               axis_label_fontsize: int):       
    
    bs = set(results.keys())
    brange = range(min(bs), max(bs) + 1)

    arr = np.full(len(brange), np.nan)

    for b, result in results.items():
        i = b - brange.start
        if (v := result.get(key)) is not None:
            arr[i] = v

    plt.bar(list(brange), arr, width=1)
    
    plt.xlabel("number of blanks", fontsize=axis_label_fontsize)
    plt.xticks(list(range(brange.start, brange.stop, 9)))
    plt.xlim(brange.start - 0.5, brange.stop - 0.5)
    plt.ylabel("reflection frequency (%)", fontsize=axis_label_fontsize)
    plt.ylim(0, 1)

    if bound is not None:
        plt.axvline(bound - 0.5, linestyle="--", color="red", linewidth=1)
    if title is not None:
        plt.title(title, fontsize=title_fontsize)


def plot_hists(
    datas: Mapping[str, PathLike],
    dpi: int = 144,
    max_ncol: int = 3,
    col_width: float = 3.5,
    row_height: float = 3.5,
    title_fontsize: int = 12,
    axis_label_fontsize: int = 11,
    bound: int | None = None,
    root: PathLike | None = None,
    show: bool = False,
    save: str | Path | None = None,
    key: str = "score",
):
    if root is not None:
        root = Path(root)

    ncol = min(max_ncol, len(datas))
    nrow = float.__ceil__(len(datas) / ncol)
    plt.figure(figsize=(ncol * col_width, nrow * row_height), dpi=dpi)
    for i, (title, path) in enumerate(datas.items(), start=1):
        if root is not None:
            path = root / path
        try:
            results = get_eval_results(path, kmap=key2blanks)
        except FileNotFoundError:
            results = None
        plt.subplot(nrow, ncol, i)
        if results:
            _plot_hist(results, key, title, bound, title_fontsize, axis_label_fontsize)
        else:
            plt.title(title + "\nNO RESULT FOUND", fontsize=title_fontsize)
    plt.tight_layout()
    if save:
        if isinstance(save, str):
            save = Path(save)
        save.parent.mkdir(parents=True, exist_ok=True)
        plt.savefig(save)
    if show:
        plt.show()
