"""App for building datasets"""

import typer

from pathlib import Path


from build_datasets.synthetic.make_synthetic import make_synthetic_dataset
from build_datasets.two_community import make_two_community_dataset

app = typer.Typer()


synth_node_min = typer.Option(20, help="Min nodes in synthetic graphs")
synth_node_max = typer.Option(160, help="Max nodes in synthetic graphs")

synth_max_each_class = typer.Option(
    500, help="Max number of each graph type. Set to 0 for no limit"
)
synth_file_name = typer.Option("synthetic.pkl", help="File name for saved dataset")
synth_save_dir = typer.Option("raw_datasets", help="Folder to save dataset in")
synth_overwrite_ok = typer.Option(
    False, help="If a file with the given name exists, whether overwriting it is ok"
)
synth_seed = typer.Option(0, help="Random seed")


@app.command()
def build_synthetic_dataset(
    node_min: int = synth_node_min,
    node_max: int = synth_node_max,
    max_each_class: int = synth_max_each_class,
    save_folder: Path = synth_save_dir,
    file_name: str = synth_file_name,
    overwrite: bool = synth_overwrite_ok,
    seed: int = synth_seed,
):
    """Make the synthetic dataset and save as pickle file"""

    make_synthetic_dataset(
        node_min=node_min,
        node_max=node_max,
        max_each_class=max_each_class,
        save_folder=save_folder,
        file_name=file_name,
        overwrite_if_exists=overwrite,
        seed=seed,
    )


@app.command()
def build_two_community_dataset():
    """Make the two-community dataset and save as pickle file"""
    
    print("Making two-community dataset")
    make_two_community_dataset("raw_datasets", "two_community.pkl")
    print("Done. Saved dataset to `raw_datasets/two_community.pkl`")


if __name__ == "__main__":
    app()
